import * as pgp from 'openpgp';
import * as readlineSync from 'readline-sync';
class PGPKeyStore {
    constructor(passphrase, pkArmor, pubkActiveArmor, pubkTrustedArmor, pubkEncryptArmor, onload = (ks) => { }) {
        this.pubk = {
            active: [],
            trusted: [],
            encrypt: [],
        };
        this.loads = 0x00;
        this.loadsTarget = 0x0f;
        this._readKey(pkArmor, undefined, 1, passphrase);
        this._readKey(pubkActiveArmor, 'active', 2);
        this._readKey(pubkTrustedArmor, 'trusted', 4);
        this._readKey(pubkEncryptArmor, 'encrypt', 8);
        this.onload = onload;
    }
    _readKey(a, x, n, pass) {
        pgp.key.readArmored(a).then((k) => {
            if (pass !== undefined) {
                this.pk = k.keys[0];
                this.pk.decrypt(pass).then(() => {
                    this.fingerprint = this.pk.getFingerprint();
                    console.log('private key (sign)', this.fingerprint);
                    this._registerLoad(n);
                });
            }
            else {
                this.pubk[x] = k.keys;
                k.keys.forEach((pubk) => {
                    console.log('public key (' + x + ')', pubk.getFingerprint());
                });
                this._registerLoad(n);
            }
        });
    }
    _registerLoad(b) {
        this.loads |= b;
        if (this.loads == this.loadsTarget) {
            this.onload(this);
        }
    }
    getTrustedKeys() {
        return this.pubk['trusted'];
    }
    getTrustedActiveKeys() {
        return this.pubk['active'];
    }
    getEncryptKeys() {
        return this.pubk['encrypt'];
    }
    getPrivateKey() {
        return this.pk;
    }
    getFingerprint() {
        return this.fingerprint;
    }
}
class PGPSigner {
    constructor(keyStore) {
        this.engine = 'pgp';
        this.algo = 'sha256';
        this.keyStore = keyStore;
        this.onsign = (string) => { };
        this.onverify = (boolean) => { };
    }
    fingerprint() {
        return this.keyStore.getFingerprint();
    }
    prepare(material) {
        this.dgst = material.digest();
        return true;
    }
    verify(digest, signature) {
        pgp.signature.readArmored(signature.data).then((s) => {
            const opts = {
                message: pgp.cleartext.fromText(digest),
                publicKeys: this.keyStore.getTrustedKeys(),
                signature: s,
            };
            pgp.verify(opts).then((v) => {
                let i = 0;
                for (i = 0; i < v.signatures.length; i++) {
                    const s = v.signatures[i];
                    if (s.valid) {
                        this.onverify(s);
                        return;
                    }
                }
                console.error('checked ' + i + ' signature(s) but none valid');
                this.onverify(false);
            });
        }).catch((e) => {
            console.error(e);
            this.onverify(false);
        });
    }
    sign(digest) {
        const m = pgp.cleartext.fromText(digest);
        const pk = this.keyStore.getPrivateKey();
        if (!pk.isDecrypted()) {
            let password;
            if (typeof window !== "undefined") {
                password = window.prompt('password');
            }
            else {
                password = readlineSync.question('Enter your private key password! ', { hideEchoBack: true });
            }
            pk.decrypt(password).catch(e => console.error(e.message));
        }
        const opts = {
            message: m,
            privateKeys: [pk],
            detached: true,
        };
        pgp.sign(opts).then((s) => {
            this.signature = {
                engine: this.engine,
                algo: this.algo,
                data: s.signature,
                // TODO: fix for browser later
                digest: digest,
            };
            this.onsign(this.signature);
        }).catch((e) => console.error(e.message));
    }
}
export { PGPSigner, PGPKeyStore, };
//# sourceMappingURL=auth.js.map