"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getScale = getScale;
exports.getQuantizeScale = getQuantizeScale;
exports.getLinearScale = getLinearScale;
exports.getQuantileScale = getQuantileScale;
exports.getOrdinalScale = getOrdinalScale;
exports.quantizeScale = quantizeScale;
exports.linearScale = linearScale;
exports.unique = unique;
exports.getLinearDomain = getLinearDomain;
exports.getQuantileDomain = getQuantileDomain;
exports.getOrdinalDomain = getOrdinalDomain;
exports.getScaleDomain = getScaleDomain;
exports.clamp = clamp;
exports.getScaleFunctionByScaleType = getScaleFunctionByScaleType;

var _core = require("@deck.gl/core");

function getScale(domain, range, scaleFunction) {
  var scale = scaleFunction;

  scale.domain = function () {
    return domain;
  };

  scale.range = function () {
    return range;
  };

  return scale;
}

function getQuantizeScale(domain, range) {
  var scaleFunction = function scaleFunction(value) {
    return quantizeScale(domain, range, value);
  };

  return getScale(domain, range, scaleFunction);
}

function getLinearScale(domain, range) {
  var scaleFunction = function scaleFunction(value) {
    return linearScale(domain, range, value);
  };

  return getScale(domain, range, scaleFunction);
}

function getQuantileScale(domain, range) {
  var sortedDomain = domain.sort(ascending);
  var i = 0;
  var n = Math.max(1, range.length);
  var thresholds = new Array(n - 1);

  while (++i < n) {
    thresholds[i - 1] = threshold(sortedDomain, i / n);
  }

  var scaleFunction = function scaleFunction(value) {
    return thresholdsScale(thresholds, range, value);
  };

  scaleFunction.thresholds = function () {
    return thresholds;
  };

  return getScale(domain, range, scaleFunction);
}

function ascending(a, b) {
  return a - b;
}

function threshold(domain, fraction) {
  var domainLength = domain.length;

  if (fraction <= 0 || domainLength < 2) {
    return domain[0];
  }

  if (fraction >= 1) {
    return domain[domainLength - 1];
  }

  var domainFraction = (domainLength - 1) * fraction;
  var lowIndex = Math.floor(domainFraction);
  var low = domain[lowIndex];
  var high = domain[lowIndex + 1];
  return low + (high - low) * (domainFraction - lowIndex);
}

function bisectRight(a, x) {
  var lo = 0;
  var hi = a.length;

  while (lo < hi) {
    var mid = lo + hi >>> 1;

    if (ascending(a[mid], x) > 0) {
      hi = mid;
    } else {
      lo = mid + 1;
    }
  }

  return lo;
}

function thresholdsScale(thresholds, range, value) {
  return range[bisectRight(thresholds, value)];
}

function ordinalScale(domain, domainMap, range, value) {
  var key = "".concat(value);
  var d = domainMap.get(key);

  if (d === undefined) {
    d = domain.push(value);
    domainMap.set(key, d);
  }

  return range[(d - 1) % range.length];
}

function getOrdinalScale(domain, range) {
  var domainMap = new Map();
  var uniqueDomain = [];
  var _iteratorNormalCompletion = true;
  var _didIteratorError = false;
  var _iteratorError = undefined;

  try {
    for (var _iterator = domain[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
      var d = _step.value;
      var key = "".concat(d);

      if (!domainMap.has(key)) {
        domainMap.set(key, uniqueDomain.push(d));
      }
    }
  } catch (err) {
    _didIteratorError = true;
    _iteratorError = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion && _iterator["return"] != null) {
        _iterator["return"]();
      }
    } finally {
      if (_didIteratorError) {
        throw _iteratorError;
      }
    }
  }

  var scaleFunction = function scaleFunction(value) {
    return ordinalScale(uniqueDomain, domainMap, range, value);
  };

  return getScale(domain, range, scaleFunction);
}

function quantizeScale(domain, range, value) {
  var domainRange = domain[1] - domain[0];

  if (domainRange <= 0) {
    _core.log.warn('quantizeScale: invalid domain, returning range[0]')();

    return range[0];
  }

  var step = domainRange / range.length;
  var idx = Math.floor((value - domain[0]) / step);
  var clampIdx = Math.max(Math.min(idx, range.length - 1), 0);
  return range[clampIdx];
}

function linearScale(domain, range, value) {
  return (value - domain[0]) / (domain[1] - domain[0]) * (range[1] - range[0]) + range[0];
}

function notNullOrUndefined(d) {
  return d !== undefined && d !== null;
}

function unique(values) {
  var results = [];
  values.forEach(function (v) {
    if (!results.includes(v) && notNullOrUndefined(v)) {
      results.push(v);
    }
  });
  return results;
}

function getTruthyValues(data, valueAccessor) {
  var values = typeof valueAccessor === 'function' ? data.map(valueAccessor) : data;
  return values.filter(notNullOrUndefined);
}

function getLinearDomain(data, valueAccessor) {
  var sorted = getTruthyValues(data, valueAccessor).sort();
  return sorted.length ? [sorted[0], sorted[sorted.length - 1]] : [0, 0];
}

function getQuantileDomain(data, valueAccessor) {
  return getTruthyValues(data, valueAccessor);
}

function getOrdinalDomain(data, valueAccessor) {
  return unique(getTruthyValues(data, valueAccessor));
}

function getScaleDomain(scaleType, data, valueAccessor) {
  switch (scaleType) {
    case 'quantize':
    case 'linear':
      return getLinearDomain(data, valueAccessor);

    case 'quantile':
      return getQuantileDomain(data, valueAccessor);

    case 'ordinal':
      return getOrdinalDomain(data, valueAccessor);

    default:
      return getLinearDomain(data, valueAccessor);
  }
}

function clamp(value, min, max) {
  return Math.max(min, Math.min(max, value));
}

function getScaleFunctionByScaleType(scaleType) {
  switch (scaleType) {
    case 'quantize':
      return getQuantizeScale;

    case 'linear':
      return getLinearScale;

    case 'quantile':
      return getQuantileScale;

    case 'ordinal':
      return getOrdinalScale;

    default:
      return getQuantizeScale;
  }
}
//# sourceMappingURL=scale-utils.js.map