import { log, COORDINATE_SYSTEM } from '@deck.gl/core';
const R_EARTH = 6378000;

function toFinite(n) {
  return Number.isFinite(n) ? n : 0;
}

export function getBoundingBox(attributes, vertexCount) {
  const positions = attributes.positions.value;
  let yMin = Infinity;
  let yMax = -Infinity;
  let xMin = Infinity;
  let xMax = -Infinity;
  let y;
  let x;

  for (let i = 0; i < vertexCount; i++) {
    x = positions[i * 3];
    y = positions[i * 3 + 1];
    yMin = y < yMin ? y : yMin;
    yMax = y > yMax ? y : yMax;
    xMin = x < xMin ? x : xMin;
    xMax = x > xMax ? x : xMax;
  }

  const boundingBox = {
    xMin: toFinite(xMin),
    xMax: toFinite(xMax),
    yMin: toFinite(yMin),
    yMax: toFinite(yMax)
  };
  return boundingBox;
}

function getTranslation(boundingBox, gridOffset, coordinateSystem, viewport) {
  const {
    width,
    height
  } = viewport;
  const worldOrigin = coordinateSystem === COORDINATE_SYSTEM.CARTESIAN ? [-width / 2, -height / 2] : [-180, -90];
  log.assert([COORDINATE_SYSTEM.CARTESIAN, COORDINATE_SYSTEM.LNGLAT, COORDINATE_SYSTEM.DEFAULT].includes(coordinateSystem));
  const {
    xMin,
    yMin
  } = boundingBox;
  return [-1 * (alignToCell(xMin - worldOrigin[0], gridOffset.xOffset) + worldOrigin[0]), -1 * (alignToCell(yMin - worldOrigin[1], gridOffset.yOffset) + worldOrigin[1])];
}

export function alignToCell(inValue, cellSize) {
  const sign = inValue < 0 ? -1 : 1;
  let value = sign < 0 ? Math.abs(inValue) + cellSize : Math.abs(inValue);
  value = Math.floor(value / cellSize) * cellSize;
  return value * sign;
}
export function getGridOffset(boundingBox, cellSize, convertToMeters = true) {
  if (!convertToMeters) {
    return {
      xOffset: cellSize,
      yOffset: cellSize
    };
  }

  const {
    yMin,
    yMax
  } = boundingBox;
  const centerLat = (yMin + yMax) / 2;
  return calculateGridLatLonOffset(cellSize, centerLat);
}
export function getGridParams(boundingBox, cellSize, viewport, coordinateSystem) {
  const gridOffset = getGridOffset(boundingBox, cellSize, coordinateSystem !== COORDINATE_SYSTEM.CARTESIAN);
  const translation = getTranslation(boundingBox, gridOffset, coordinateSystem, viewport);
  const {
    xMin,
    yMin,
    xMax,
    yMax
  } = boundingBox;
  const width = xMax - xMin + gridOffset.xOffset;
  const height = yMax - yMin + gridOffset.yOffset;
  const numCol = Math.ceil(width / gridOffset.xOffset);
  const numRow = Math.ceil(height / gridOffset.yOffset);
  return {
    gridOffset,
    translation,
    width,
    height,
    numCol,
    numRow
  };
}

function calculateGridLatLonOffset(cellSize, latitude) {
  const yOffset = calculateLatOffset(cellSize);
  const xOffset = calculateLonOffset(latitude, cellSize);
  return {
    yOffset,
    xOffset
  };
}

function calculateLatOffset(dy) {
  return dy / R_EARTH * (180 / Math.PI);
}

function calculateLonOffset(lat, dx) {
  return dx / R_EARTH * (180 / Math.PI) / Math.cos(lat * Math.PI / 180);
}
//# sourceMappingURL=grid-aggregation-utils.js.map