import { log } from '@deck.gl/core';
import GPUGridAggregator from '../utils/gpu-grid-aggregation/gpu-grid-aggregator';
import { AGGREGATION_OPERATION, getValueFunc } from '../utils/aggregation-operation-utils';
import ScreenGridCellLayer from './screen-grid-cell-layer';
import GridAggregationLayer from '../grid-aggregation-layer';
import { getFloatTexture } from '../utils/resource-utils.js';
const defaultProps = Object.assign({}, ScreenGridCellLayer.defaultProps, {
  getPosition: {
    type: 'accessor',
    value: d => d.position
  },
  getWeight: {
    type: 'accessor',
    value: d => 1
  },
  gpuAggregation: true,
  aggregation: 'SUM'
});
const POSITION_ATTRIBUTE_NAME = 'positions';
const DIMENSIONS = {
  data: {
    props: ['cellSizePixels']
  },
  weights: {
    props: ['aggregation'],
    accessors: ['getWeight']
  }
};
export default class ScreenGridLayer extends GridAggregationLayer {
  initializeState() {
    const {
      gl
    } = this.context;

    if (!ScreenGridCellLayer.isSupported(gl)) {
      this.setState({
        supported: false
      });
      log.error("ScreenGridLayer: ".concat(this.id, " is not supported on this browser"))();
      return;
    }

    super.initializeState({
      dimensions: DIMENSIONS,
      getCellSize: props => props.cellSizePixels
    });
    const weights = {
      count: {
        size: 1,
        operation: AGGREGATION_OPERATION.SUM,
        needMax: true,
        maxTexture: getFloatTexture(gl, {
          id: "".concat(this.id, "-max-texture")
        })
      }
    };
    this.setState({
      supported: true,
      projectPoints: true,
      weights,
      subLayerData: {
        attributes: {}
      },
      maxTexture: weights.count.maxTexture,
      positionAttributeName: 'positions',
      posOffset: [0, 0],
      translation: [1, -1]
    });
    const attributeManager = this.getAttributeManager();
    attributeManager.add({
      [POSITION_ATTRIBUTE_NAME]: {
        size: 3,
        accessor: 'getPosition',
        type: 5130,
        fp64: this.use64bitPositions()
      },
      count: {
        size: 3,
        accessor: 'getWeight'
      }
    });
  }

  shouldUpdateState({
    changeFlags
  }) {
    return this.state.supported && changeFlags.somethingChanged;
  }

  updateState(opts) {
    super.updateState(opts);
  }

  renderLayers() {
    if (!this.state.supported) {
      return [];
    }

    const {
      maxTexture,
      numRow,
      numCol,
      weights
    } = this.state;
    const {
      updateTriggers
    } = this.props;
    const {
      aggregationBuffer
    } = weights.count;
    const CellLayerClass = this.getSubLayerClass('cells', ScreenGridCellLayer);
    return new CellLayerClass(this.props, this.getSubLayerProps({
      id: 'cell-layer',
      updateTriggers
    }), {
      data: {
        attributes: {
          instanceCounts: aggregationBuffer
        }
      },
      maxTexture,
      numInstances: numRow * numCol
    });
  }

  finalizeState() {
    super.finalizeState();
    const {
      aggregationBuffer,
      maxBuffer,
      maxTexture
    } = this.state;

    if (aggregationBuffer) {
      aggregationBuffer.delete();
    }

    if (maxBuffer) {
      maxBuffer.delete();
    }

    if (maxTexture) {
      maxTexture.delete();
    }
  }

  getPickingInfo({
    info,
    mode
  }) {
    const {
      index
    } = info;

    if (index >= 0) {
      const {
        gpuGridAggregator
      } = this.state;
      const aggregationResults = gpuGridAggregator.getData('count');
      info.object = GPUGridAggregator.getAggregationData(Object.assign({
        pixelIndex: index
      }, aggregationResults));
    }

    return info;
  }

  updateResults({
    aggregationData,
    maxData
  }) {
    const {
      count
    } = this.state.weights;
    count.aggregationData = aggregationData;
    count.aggregationBuffer.setData({
      data: aggregationData
    });
    count.maxData = maxData;
    count.maxTexture.setImageData({
      data: maxData
    });
  }

  updateAggregationState(opts) {
    const cellSize = opts.props.cellSizePixels;
    const cellSizeChanged = opts.oldProps.cellSizePixels !== cellSize;
    const {
      viewportChanged
    } = opts.changeFlags;
    let gpuAggregation = opts.props.gpuAggregation;

    if (this.state.gpuAggregation !== opts.props.gpuAggregation) {
      if (gpuAggregation && !GPUGridAggregator.isSupported(this.context.gl)) {
        log.warn('GPU Grid Aggregation not supported, falling back to CPU')();
        gpuAggregation = false;
      }
    }

    const gpuAggregationChanged = gpuAggregation !== this.state.gpuAggregation;
    this.setState({
      gpuAggregation
    });
    const positionsChanged = this.isAttributeChanged(POSITION_ATTRIBUTE_NAME);
    const {
      dimensions
    } = this.state;
    const {
      data,
      weights
    } = dimensions;
    const aggregationDataDirty = positionsChanged || gpuAggregationChanged || viewportChanged || this.isAggregationDirty(opts, {
      compareAll: gpuAggregation,
      dimension: data
    });
    const aggregationWeightsDirty = this.isAggregationDirty(opts, {
      dimension: weights
    });
    this.setState({
      aggregationDataDirty,
      aggregationWeightsDirty
    });
    const {
      viewport
    } = this.context;

    if (viewportChanged || cellSizeChanged) {
      const {
        width,
        height
      } = viewport;
      const numCol = Math.ceil(width / cellSize);
      const numRow = Math.ceil(height / cellSize);
      this.allocateResources(numRow, numCol);
      this.setState({
        scaling: [width / 2, -height / 2, 1],
        gridOffset: {
          xOffset: cellSize,
          yOffset: cellSize
        },
        width,
        height,
        numCol,
        numRow
      });
    }

    if (aggregationWeightsDirty) {
      this._updateAccessors(opts);
    }

    if (aggregationDataDirty || aggregationWeightsDirty) {
      this._resetResults();
    }
  }

  _updateAccessors(opts) {
    const {
      getWeight,
      aggregation
    } = opts.props;
    const {
      count
    } = this.state.weights;

    if (count) {
      count.getWeight = getWeight;
      count.operation = AGGREGATION_OPERATION[aggregation];
    }

    this.setState({
      getValue: getValueFunc(aggregation, getWeight)
    });
  }

  _resetResults() {
    const {
      count
    } = this.state.weights;

    if (count) {
      count.aggregationData = null;
    }
  }

}
ScreenGridLayer.layerName = 'ScreenGridLayer';
ScreenGridLayer.defaultProps = defaultProps;
//# sourceMappingURL=screen-grid-layer.js.map