const defaultGetValue = points => points.length;

import { clamp, getQuantileDomain, getOrdinalDomain } from './scale-utils';
const MAX_32_BIT_FLOAT = 3.402823466e38;

const defaultGetPoints = bin => bin.points;

const defaultGetIndex = bin => bin.index;

const ascending = (a, b) => a < b ? -1 : a > b ? 1 : a >= b ? 0 : NaN;

const defaultProps = {
  getValue: defaultGetValue,
  getPoints: defaultGetPoints,
  getIndex: defaultGetIndex,
  filterData: null
};
export default class BinSorter {
  constructor(bins = [], props = defaultProps) {
    this.aggregatedBins = this.getAggregatedBins(bins, props);

    this._updateMinMaxValues();

    this.binMap = this.getBinMap();
  }

  getAggregatedBins(bins, props) {
    const {
      getValue = defaultGetValue,
      getPoints = defaultGetPoints,
      getIndex = defaultGetIndex,
      filterData
    } = props;
    const hasFilter = typeof filterData === 'function';
    const binCount = bins.length;
    const aggregatedBins = [];
    let index = 0;

    for (let binIndex = 0; binIndex < binCount; binIndex++) {
      const bin = bins[binIndex];
      const points = getPoints(bin);
      const i = getIndex(bin);
      const filteredPoints = hasFilter ? points.filter(filterData) : points;
      bin.filteredPoints = hasFilter ? filteredPoints : null;
      const value = filteredPoints.length ? getValue(filteredPoints) : null;

      if (value !== null && value !== undefined) {
        aggregatedBins[index] = {
          i: Number.isFinite(i) ? i : binIndex,
          value,
          counts: filteredPoints.length
        };
        index++;
      }
    }

    return aggregatedBins;
  }

  _percentileToIndex(percentileRange) {
    const len = this.sortedBins.length;

    if (len < 2) {
      return [0, 0];
    }

    const [lower, upper] = percentileRange.map(n => clamp(n, 0, 100));
    const lowerIdx = Math.ceil(lower / 100 * (len - 1));
    const upperIdx = Math.floor(upper / 100 * (len - 1));
    return [lowerIdx, upperIdx];
  }

  getBinMap() {
    const binMap = {};

    for (const bin of this.aggregatedBins) {
      binMap[bin.i] = bin;
    }

    return binMap;
  }

  _updateMinMaxValues() {
    let maxCount = 0;
    let maxValue = 0;
    let minValue = MAX_32_BIT_FLOAT;
    let totalCount = 0;

    for (const x of this.aggregatedBins) {
      maxCount = maxCount > x.counts ? maxCount : x.counts;
      maxValue = maxValue > x.value ? maxValue : x.value;
      minValue = minValue < x.value ? minValue : x.value;
      totalCount += x.counts;
    }

    this.maxCount = maxCount;
    this.maxValue = maxValue;
    this.minValue = minValue;
    this.totalCount = totalCount;
  }

  getValueRange(percentileRange) {
    if (!this.sortedBins) {
      this.sortedBins = this.aggregatedBins.sort((a, b) => ascending(a.value, b.value));
    }

    if (!this.sortedBins.length) {
      return [];
    }

    let lowerIdx = 0;
    let upperIdx = this.sortedBins.length - 1;

    if (Array.isArray(percentileRange)) {
      const idxRange = this._percentileToIndex(percentileRange);

      lowerIdx = idxRange[0];
      upperIdx = idxRange[1];
    }

    return [this.sortedBins[lowerIdx].value, this.sortedBins[upperIdx].value];
  }

  getValueDomainByScale(scale, [lower = 0, upper = 100] = []) {
    if (!this.sortedBins) {
      this.sortedBins = this.aggregatedBins.sort((a, b) => ascending(a.value, b.value));
    }

    if (!this.sortedBins.length) {
      return [];
    }

    const indexEdge = this._percentileToIndex([lower, upper]);

    return this._getScaleDomain(scale, indexEdge);
  }

  _getScaleDomain(scaleType, [lowerIdx, upperIdx]) {
    const bins = this.sortedBins;

    switch (scaleType) {
      case 'quantize':
      case 'linear':
        return [bins[lowerIdx].value, bins[upperIdx].value];

      case 'quantile':
        return getQuantileDomain(bins.slice(lowerIdx, upperIdx + 1), d => d.value);

      case 'ordinal':
        return getOrdinalDomain(bins, d => d.value);

      default:
        return [bins[lowerIdx].value, bins[upperIdx].value];
    }
  }

}
//# sourceMappingURL=bin-sorter.js.map