"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.normalizeTransitionSettings = normalizeTransitionSettings;
exports.getSourceBufferAttribute = getSourceBufferAttribute;
exports.getAttributeTypeFromSize = getAttributeTypeFromSize;
exports.cycleBuffers = cycleBuffers;
exports.getAttributeBufferLength = getAttributeBufferLength;
exports.padBuffer = padBuffer;

var _arrayUtils = require("../../utils/array-utils");

var DEFAULT_TRANSITION_SETTINGS = {
  interpolation: {
    duration: 0,
    easing: function easing(t) {
      return t;
    }
  },
  spring: {
    stiffness: 0.05,
    damping: 0.5
  }
};

function normalizeTransitionSettings(userSettings, layerSettings) {
  if (!userSettings) {
    return null;
  }

  if (Number.isFinite(userSettings)) {
    userSettings = {
      duration: userSettings
    };
  }

  userSettings.type = userSettings.type || 'interpolation';
  return Object.assign({}, DEFAULT_TRANSITION_SETTINGS[userSettings.type], layerSettings, userSettings);
}

function getSourceBufferAttribute(gl, attribute) {
  var buffer = attribute.getBuffer();

  if (buffer) {
    return [attribute.getBuffer(), {
      divisor: 0,
      size: attribute.size,
      normalized: attribute.settings.normalized
    }];
  }

  return attribute.value;
}

function getAttributeTypeFromSize(size) {
  switch (size) {
    case 1:
      return 'float';

    case 2:
      return 'vec2';

    case 3:
      return 'vec3';

    case 4:
      return 'vec4';

    default:
      throw new Error("No defined attribute type for size \"".concat(size, "\""));
  }
}

function cycleBuffers(buffers) {
  buffers.push(buffers.shift());
}

function getAttributeBufferLength(attribute, numInstances) {
  var doublePrecision = attribute.doublePrecision,
      settings = attribute.settings,
      value = attribute.value,
      size = attribute.size;
  var multiplier = doublePrecision ? 2 : 1;
  return (settings.noAlloc ? value.length : numInstances * size) * multiplier;
}

function padBuffer(_ref) {
  var buffer = _ref.buffer,
      numInstances = _ref.numInstances,
      attribute = _ref.attribute,
      fromLength = _ref.fromLength,
      fromStartIndices = _ref.fromStartIndices,
      _ref$getData = _ref.getData,
      getData = _ref$getData === void 0 ? function (x) {
    return x;
  } : _ref$getData;
  var precisionMultiplier = attribute.doublePrecision ? 2 : 1;
  var size = attribute.size * precisionMultiplier;
  var byteOffset = attribute.byteOffset;
  var toStartIndices = attribute.startIndices;
  var hasStartIndices = fromStartIndices && toStartIndices;
  var toLength = getAttributeBufferLength(attribute, numInstances);
  var isConstant = attribute.state.constant;

  if (!hasStartIndices && fromLength >= toLength) {
    return;
  }

  var toData = isConstant ? attribute.value : attribute.getBuffer().getData({
    srcByteOffset: byteOffset
  });

  if (attribute.settings.normalized && !isConstant) {
    var getter = getData;

    getData = function getData(value, chunk) {
      return attribute._normalizeConstant(getter(value, chunk));
    };
  }

  var getMissingData = isConstant ? function (i, chunk) {
    return getData(toData, chunk);
  } : function (i, chunk) {
    return getData(toData.subarray(i, i + size), chunk);
  };
  var source = buffer.getData({
    length: fromLength
  });
  var data = new Float32Array(toLength);
  (0, _arrayUtils.padArray)({
    source: source,
    target: data,
    sourceStartIndices: fromStartIndices,
    targetStartIndices: toStartIndices,
    size: size,
    getData: getMissingData
  });

  if (buffer.byteLength < data.byteLength + byteOffset) {
    buffer.reallocate(data.byteLength + byteOffset);
  }

  buffer.subData({
    data: data,
    offset: byteOffset
  });
}
//# sourceMappingURL=attribute-transition-utils.js.map