import { clamp, Vector2 } from 'math.gl';
import Controller from './controller';
import ViewState from './view-state';
import LinearInterpolator from '../transitions/linear-interpolator';
import { TRANSITION_EVENTS } from './transition-manager';
import { mod } from '../utils/math-utils';
const MOVEMENT_SPEED = 50;
const DEFAULT_STATE = {
  orbitAxis: 'Z',
  rotationX: 0,
  rotationOrbit: 0,
  zoom: 0,
  target: [0, 0, 0],
  minRotationX: -90,
  maxRotationX: 90,
  minZoom: -Infinity,
  maxZoom: Infinity
};
const LINEAR_TRANSITION_PROPS = {
  transitionDuration: 300,
  transitionEasing: t => t,
  transitionInterpolator: new LinearInterpolator(['target', 'zoom', 'rotationX', 'rotationOrbit']),
  transitionInterruption: TRANSITION_EVENTS.BREAK
};

const zoom2Scale = zoom => Math.pow(2, zoom);

export class OrbitState extends ViewState {
  constructor({
    makeViewport,
    width,
    height,
    orbitAxis = DEFAULT_STATE.orbitAxis,
    rotationX = DEFAULT_STATE.rotationX,
    rotationOrbit = DEFAULT_STATE.rotationOrbit,
    target = DEFAULT_STATE.target,
    zoom = DEFAULT_STATE.zoom,
    minRotationX = DEFAULT_STATE.minRotationX,
    maxRotationX = DEFAULT_STATE.maxRotationX,
    minZoom = DEFAULT_STATE.minZoom,
    maxZoom = DEFAULT_STATE.maxZoom,
    startPanPosition,
    startTarget,
    startRotationX,
    startRotationOrbit,
    startZoomPosition,
    startZoom
  }) {
    super({
      width,
      height,
      orbitAxis,
      rotationX,
      rotationOrbit,
      target,
      zoom,
      minRotationX,
      maxRotationX,
      minZoom,
      maxZoom
    });
    this._interactiveState = {
      startPanPosition,
      startTarget,
      startRotationX,
      startRotationOrbit,
      startZoomPosition,
      startZoom
    };
    this.makeViewport = makeViewport;
  }

  getViewportProps() {
    return this._viewportProps;
  }

  getInteractiveState() {
    return this._interactiveState;
  }

  panStart({
    pos
  }) {
    const {
      target
    } = this._viewportProps;
    return this._getUpdatedState({
      startPanPosition: pos,
      startTarget: target
    });
  }

  pan({
    pos,
    startPos
  }) {
    const {
      startPanPosition,
      startTarget
    } = this._interactiveState;
    const delta = new Vector2(pos).subtract(startPanPosition);
    return this._getUpdatedState({
      target: this._calculateNewTarget({
        startTarget,
        pixelOffset: delta
      })
    });
  }

  panEnd() {
    return this._getUpdatedState({
      startPanPosition: null,
      startTarget: null
    });
  }

  rotateStart({
    pos
  }) {
    return this._getUpdatedState({
      startRotationX: this._viewportProps.rotationX,
      startRotationOrbit: this._viewportProps.rotationOrbit
    });
  }

  rotate({
    deltaScaleX,
    deltaScaleY
  }) {
    const {
      startRotationX,
      startRotationOrbit
    } = this._interactiveState;

    if (!Number.isFinite(startRotationX) || !Number.isFinite(startRotationOrbit)) {
      return this;
    }

    if (startRotationX < -90 || startRotationX > 90) {
      deltaScaleX *= -1;
    }

    return this._getUpdatedState({
      rotationX: startRotationX + deltaScaleY * 180,
      rotationOrbit: startRotationOrbit + deltaScaleX * 180,
      isRotating: true
    });
  }

  rotateEnd() {
    return this._getUpdatedState({
      startRotationX: null,
      startRotationOrbit: null
    });
  }

  shortestPathFrom(viewState) {
    const fromProps = viewState.getViewportProps();
    const props = Object.assign({}, this._viewportProps);
    const {
      rotationOrbit
    } = props;

    if (Math.abs(rotationOrbit - fromProps.rotationOrbit) > 180) {
      props.rotationOrbit = rotationOrbit < 0 ? rotationOrbit + 360 : rotationOrbit - 360;
    }

    return props;
  }

  zoomStart({
    pos
  }) {
    return this._getUpdatedState({
      startZoomPosition: pos,
      startTarget: this._viewportProps.target,
      startZoom: this._viewportProps.zoom
    });
  }

  zoom({
    pos,
    startPos,
    scale
  }) {
    const {
      zoom,
      width,
      height,
      target
    } = this._viewportProps;
    let {
      startZoom,
      startZoomPosition,
      startTarget
    } = this._interactiveState;

    if (!Number.isFinite(startZoom)) {
      startZoom = zoom;
      startTarget = target;
      startZoomPosition = startPos || pos;
    }

    const newZoom = this._calculateNewZoom({
      scale,
      startZoom
    });

    const startScale = zoom2Scale(startZoom);
    const newScale = zoom2Scale(newZoom);
    const dX = (width / 2 - startZoomPosition[0]) * (newScale / startScale - 1);
    const dY = (height / 2 - startZoomPosition[1]) * (newScale / startScale - 1);
    return this._getUpdatedState({
      zoom: newZoom,
      target: this._calculateNewTarget({
        startTarget,
        zoom: newZoom,
        pixelOffset: [dX, dY]
      })
    });
  }

  zoomEnd() {
    return this._getUpdatedState({
      startZoomPosition: null,
      startTarget: null,
      startZoom: null
    });
  }

  zoomIn() {
    return this._getUpdatedState({
      zoom: this._calculateNewZoom({
        scale: 2
      })
    });
  }

  zoomOut() {
    return this._getUpdatedState({
      zoom: this._calculateNewZoom({
        scale: 0.5
      })
    });
  }

  moveLeft() {
    const pixelOffset = [-MOVEMENT_SPEED, 0];
    return this._getUpdatedState({
      target: this._calculateNewTarget({
        pixelOffset
      })
    });
  }

  moveRight() {
    const pixelOffset = [MOVEMENT_SPEED, 0];
    return this._getUpdatedState({
      target: this._calculateNewTarget({
        pixelOffset
      })
    });
  }

  moveUp() {
    const pixelOffset = [0, -MOVEMENT_SPEED];
    return this._getUpdatedState({
      target: this._calculateNewTarget({
        pixelOffset
      })
    });
  }

  moveDown() {
    const pixelOffset = [0, MOVEMENT_SPEED];
    return this._getUpdatedState({
      target: this._calculateNewTarget({
        pixelOffset
      })
    });
  }

  rotateLeft() {
    return this._getUpdatedState({
      rotationOrbit: this._viewportProps.rotationOrbit - 15
    });
  }

  rotateRight() {
    return this._getUpdatedState({
      rotationOrbit: this._viewportProps.rotationOrbit + 15
    });
  }

  rotateUp() {
    return this._getUpdatedState({
      rotationX: this._viewportProps.rotationX - 10
    });
  }

  rotateDown() {
    return this._getUpdatedState({
      rotationX: this._viewportProps.rotationX + 10
    });
  }

  _calculateNewZoom({
    scale,
    startZoom
  }) {
    const {
      maxZoom,
      minZoom
    } = this._viewportProps;

    if (!Number.isFinite(startZoom)) {
      startZoom = this._viewportProps.zoom;
    }

    const zoom = startZoom + Math.log2(scale);
    return clamp(zoom, minZoom, maxZoom);
  }

  _calculateNewTarget({
    startTarget,
    zoom,
    pixelOffset
  }) {
    const viewportProps = Object.assign({}, this._viewportProps);

    if (Number.isFinite(zoom)) {
      viewportProps.zoom = zoom;
    }

    if (startTarget) {
      viewportProps.target = startTarget;
    }

    const viewport = this.makeViewport(viewportProps);
    const center = viewport.project(viewportProps.target);
    return viewport.unproject([center[0] - pixelOffset[0], center[1] - pixelOffset[1], center[2]]);
  }

  _getUpdatedState(newProps) {
    return new OrbitState(Object.assign({}, this._viewportProps, this._interactiveState, newProps));
  }

  _applyConstraints(props) {
    const {
      maxZoom,
      minZoom,
      zoom,
      maxRotationX,
      minRotationX,
      rotationOrbit
    } = props;
    props.zoom = clamp(zoom, minZoom, maxZoom);
    props.rotationX = clamp(props.rotationX, minRotationX, maxRotationX);

    if (rotationOrbit < -180 || rotationOrbit > 180) {
      props.rotationOrbit = mod(rotationOrbit + 180, 360) - 180;
    }

    return props;
  }

}
export default class OrbitController extends Controller {
  constructor(props) {
    super(OrbitState, props);
  }

  _getTransitionProps() {
    return LINEAR_TRANSITION_PROPS;
  }

}
//# sourceMappingURL=orbit-controller.js.map