import Attribute from './attribute';
import log from '../../utils/log';
import debug from '../../debug';
import AttributeTransitionManager from './attribute-transition-manager';
const TRACE_INVALIDATE = 'attributeManager.invalidate';
const TRACE_UPDATE_START = 'attributeManager.updateStart';
const TRACE_UPDATE_END = 'attributeManager.updateEnd';
const TRACE_ATTRIBUTE_UPDATE_START = 'attribute.updateStart';
const TRACE_ATTRIBUTE_ALLOCATE = 'attribute.allocate';
const TRACE_ATTRIBUTE_UPDATE_END = 'attribute.updateEnd';
export default class AttributeManager {
  constructor(gl, {
    id = 'attribute-manager',
    stats,
    timeline
  } = {}) {
    this.id = id;
    this.gl = gl;
    this.attributes = {};
    this.updateTriggers = {};
    this.accessors = {};
    this.needsRedraw = true;
    this.userData = {};
    this.stats = stats;
    this.attributeTransitionManager = new AttributeTransitionManager(gl, {
      id: "".concat(id, "-transitions"),
      timeline
    });
    Object.seal(this);
  }

  finalize() {
    for (const attributeName in this.attributes) {
      this.attributes[attributeName].delete();
    }

    this.attributeTransitionManager.finalize();
  }

  getNeedsRedraw(opts = {
    clearRedrawFlags: false
  }) {
    const redraw = this.needsRedraw;
    this.needsRedraw = this.needsRedraw && !opts.clearRedrawFlags;
    return redraw && this.id;
  }

  setNeedsRedraw(redraw = true) {
    this.needsRedraw = true;
    return this;
  }

  add(attributes, updaters) {
    this._add(attributes, updaters);
  }

  addInstanced(attributes, updaters) {
    this._add(attributes, updaters, {
      instanced: 1
    });
  }

  remove(attributeNameArray) {
    for (let i = 0; i < attributeNameArray.length; i++) {
      const name = attributeNameArray[i];

      if (this.attributes[name] !== undefined) {
        this.attributes[name].delete();
        delete this.attributes[name];
      }
    }
  }

  invalidate(triggerName, dataRange) {
    const invalidatedAttributes = this._invalidateTrigger(triggerName, dataRange);

    debug(TRACE_INVALIDATE, this, triggerName, invalidatedAttributes);
  }

  invalidateAll(dataRange) {
    for (const attributeName in this.attributes) {
      this.attributes[attributeName].setNeedsUpdate(attributeName, dataRange);
    }

    debug(TRACE_INVALIDATE, this, 'all');
  }

  update({
    data,
    numInstances,
    startIndices = null,
    transitions,
    props = {},
    buffers = {},
    context = {}
  } = {}) {
    let updated = false;
    debug(TRACE_UPDATE_START, this);

    if (this.stats) {
      this.stats.get('Update Attributes').timeStart();
    }

    for (const attributeName in this.attributes) {
      const attribute = this.attributes[attributeName];
      const accessorName = attribute.settings.accessor;
      attribute.startIndices = startIndices;

      if (props[attributeName]) {
        log.removed("props.".concat(attributeName), "data.attributes.".concat(attributeName))();
      }

      if (attribute.setExternalBuffer(buffers[attributeName])) {} else if (attribute.setBinaryValue(buffers[accessorName], data.startIndices)) {} else if (!buffers[accessorName] && attribute.setConstantValue(props[accessorName])) {} else if (attribute.needsUpdate()) {
        updated = true;

        this._updateAttribute({
          attribute,
          numInstances,
          data,
          props,
          context
        });
      }

      this.needsRedraw |= attribute.needsRedraw();
    }

    if (updated) {
      debug(TRACE_UPDATE_END, this, numInstances);
    }

    if (this.stats) {
      this.stats.get('Update Attributes').timeEnd();
    }

    this.attributeTransitionManager.update({
      attributes: this.attributes,
      numInstances,
      transitions
    });
  }

  updateTransition() {
    const {
      attributeTransitionManager
    } = this;
    const transitionUpdated = attributeTransitionManager.run();
    this.needsRedraw = this.needsRedraw || transitionUpdated;
    return transitionUpdated;
  }

  getAttributes() {
    return this.attributes;
  }

  getChangedAttributes(opts = {
    clearChangedFlags: false
  }) {
    const {
      attributes,
      attributeTransitionManager
    } = this;
    const changedAttributes = Object.assign({}, attributeTransitionManager.getAttributes());

    for (const attributeName in attributes) {
      const attribute = attributes[attributeName];

      if (attribute.needsRedraw(opts) && !attributeTransitionManager.hasAttribute(attributeName)) {
        changedAttributes[attributeName] = attribute;
      }
    }

    return changedAttributes;
  }

  getShaderAttributes(attributes, excludeAttributes = {}) {
    if (!attributes) {
      attributes = this.getAttributes();
    }

    const shaderAttributes = {};

    for (const attributeName in attributes) {
      if (!excludeAttributes[attributeName]) {
        Object.assign(shaderAttributes, attributes[attributeName].getShaderAttributes());
      }
    }

    return shaderAttributes;
  }

  getAccessors() {
    return this.updateTriggers;
  }

  _add(attributes, updaters, extraProps = {}) {
    if (updaters) {
      log.warn('AttributeManager.add({updaters}) - updater map no longer supported')();
    }

    const newAttributes = {};

    for (const attributeName in attributes) {
      const attribute = attributes[attributeName];

      const newAttribute = this._createAttribute(attributeName, attribute, extraProps);

      newAttributes[attributeName] = newAttribute;
    }

    Object.assign(this.attributes, newAttributes);

    this._mapUpdateTriggersToAttributes();
  }

  _createAttribute(name, attribute, extraProps) {
    const props = {
      id: name,
      constant: attribute.constant || false,
      isIndexed: attribute.isIndexed || attribute.elements,
      size: attribute.elements && 1 || attribute.size,
      value: attribute.value || null,
      divisor: attribute.instanced || extraProps.instanced ? 1 : attribute.divisor
    };
    return new Attribute(this.gl, Object.assign({}, attribute, props));
  }

  _mapUpdateTriggersToAttributes() {
    const triggers = {};

    for (const attributeName in this.attributes) {
      const attribute = this.attributes[attributeName];
      attribute.getUpdateTriggers().forEach(triggerName => {
        if (!triggers[triggerName]) {
          triggers[triggerName] = [];
        }

        triggers[triggerName].push(attributeName);
      });
    }

    this.updateTriggers = triggers;
  }

  _invalidateTrigger(triggerName, dataRange) {
    const {
      attributes,
      updateTriggers
    } = this;
    const invalidatedAttributes = updateTriggers[triggerName];

    if (invalidatedAttributes) {
      invalidatedAttributes.forEach(name => {
        const attribute = attributes[name];

        if (attribute) {
          attribute.setNeedsUpdate(attribute.id, dataRange);
        }
      });
    }

    return invalidatedAttributes;
  }

  _updateAttribute(opts) {
    const {
      attribute,
      numInstances
    } = opts;
    debug(TRACE_ATTRIBUTE_UPDATE_START, attribute);

    if (attribute.allocate(numInstances)) {
      debug(TRACE_ATTRIBUTE_ALLOCATE, attribute, numInstances);
    }

    const updated = attribute.updateBuffer(opts);

    if (updated) {
      this.needsRedraw = true;
      debug(TRACE_ATTRIBUTE_UPDATE_END, attribute, numInstances);
    }
  }

}
//# sourceMappingURL=attribute-manager.js.map