import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); if (enumerableOnly) symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; }); keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; if (i % 2) { ownKeys(Object(source), true).forEach(function (key) { _defineProperty(target, key, source[key]); }); } else if (Object.getOwnPropertyDescriptors) { Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)); } else { ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } } return target; }

import assert from '../utils/assert';
import { deepEqual } from '../utils/deep-equal';
import log from '../utils/log';
import { flatten } from '../utils/flatten';
export default class ViewManager {
  constructor(props = {}) {
    this.views = [];
    this.width = 100;
    this.height = 100;
    this.viewState = {};
    this.controllers = {};
    this.timeline = props.timeline;
    this._viewports = [];
    this._viewportMap = {};
    this._isUpdating = false;
    this._needsRedraw = 'Initial render';
    this._needsUpdate = true;
    this._eventManager = props.eventManager;
    this._eventCallbacks = {
      onViewStateChange: props.onViewStateChange,
      onInteractiveStateChange: props.onInteractiveStateChange
    };
    Object.seal(this);
    this.setProps(props);
  }

  finalize() {
    for (const key in this.controllers) {
      if (this.controllers[key]) {
        this.controllers[key].finalize();
      }
    }

    this.controllers = {};
  }

  needsRedraw(opts = {
    clearRedrawFlags: false
  }) {
    const redraw = this._needsRedraw;

    if (opts.clearRedrawFlags) {
      this._needsRedraw = false;
    }

    return redraw;
  }

  setNeedsUpdate(reason) {
    this._needsUpdate = this._needsUpdate || reason;
    this._needsRedraw = this._needsRedraw || reason;
  }

  updateViewStates() {
    for (const viewId in this.controllers) {
      const controller = this.controllers[viewId];

      if (controller) {
        controller.updateTransition();
      }
    }
  }

  getViewports(rect) {
    if (rect) {
      return this._viewports.filter(viewport => viewport.containsPixel(rect));
    }

    return this._viewports;
  }

  getViews() {
    const viewMap = {};
    this.views.forEach(view => {
      viewMap[view.id] = view;
    });
    return viewMap;
  }

  getView(viewOrViewId) {
    return typeof viewOrViewId === 'string' ? this.views.find(view => view.id === viewOrViewId) : viewOrViewId;
  }

  getViewState(viewId) {
    const view = this.getView(viewId);
    const viewState = view && this.viewState[view.getViewStateId()] || this.viewState;
    return view ? view.filterViewState(viewState) : viewState;
  }

  getViewport(viewId) {
    return this._viewportMap[viewId];
  }

  unproject(xyz, opts) {
    const viewports = this.getViewports();
    const pixel = {
      x: xyz[0],
      y: xyz[1]
    };

    for (let i = viewports.length - 1; i >= 0; --i) {
      const viewport = viewports[i];

      if (viewport.containsPixel(pixel)) {
        const p = xyz.slice();
        p[0] -= viewport.x;
        p[1] -= viewport.y;
        return viewport.unproject(p, opts);
      }
    }

    return null;
  }

  setProps(props) {
    if ('views' in props) {
      this._setViews(props.views);
    }

    if ('viewState' in props) {
      this._setViewState(props.viewState);
    }

    if ('width' in props || 'height' in props) {
      this._setSize(props.width, props.height);
    }

    if (!this._isUpdating) {
      this._update();
    }
  }

  _update() {
    this._isUpdating = true;

    if (this._needsUpdate) {
      this._needsUpdate = false;

      this._rebuildViewports();
    }

    if (this._needsUpdate) {
      this._needsUpdate = false;

      this._rebuildViewports();
    }

    this._isUpdating = false;
  }

  _setSize(width, height) {
    assert(Number.isFinite(width) && Number.isFinite(height));

    if (width !== this.width || height !== this.height) {
      this.width = width;
      this.height = height;
      this.setNeedsUpdate('Size changed');
    }
  }

  _setViews(views) {
    views = flatten(views, Boolean);

    const viewsChanged = this._diffViews(views, this.views);

    if (viewsChanged) {
      this.setNeedsUpdate('views changed');
    }

    this.views = views;
  }

  _setViewState(viewState) {
    if (viewState) {
      const viewStateChanged = !deepEqual(viewState, this.viewState);

      if (viewStateChanged) {
        this.setNeedsUpdate('viewState changed');
      }

      this.viewState = viewState;
    } else {
      log.warn('missing `viewState` or `initialViewState`')();
    }
  }

  _onViewStateChange(viewId, event) {
    event.viewId = viewId;

    this._eventCallbacks.onViewStateChange(event);
  }

  _createController(view, props) {
    const Controller = props.type;
    const controller = new Controller(_objectSpread({
      timeline: this.timeline,
      eventManager: this._eventManager,
      onViewStateChange: this._onViewStateChange.bind(this, props.id),
      onStateChange: this._eventCallbacks.onInteractiveStateChange,
      makeViewport: view._getViewport.bind(view)
    }, props));
    return controller;
  }

  _updateController(view, viewState, viewport, controller) {
    let controllerProps = view.controller;

    if (controllerProps) {
      controllerProps = _objectSpread({}, viewState, {}, view.props, {}, controllerProps, {
        id: view.id,
        x: viewport.x,
        y: viewport.y,
        width: viewport.width,
        height: viewport.height
      });

      if (controller) {
        controller.setProps(controllerProps);
      } else {
        controller = this._createController(view, controllerProps);
      }

      return controller;
    }

    return null;
  }

  _rebuildViewports() {
    const {
      width,
      height,
      views
    } = this;
    const oldControllers = this.controllers;
    this._viewports = [];
    this.controllers = {};

    for (let i = views.length; i--;) {
      const view = views[i];
      const viewState = this.getViewState(view);
      const viewport = view.makeViewport({
        width,
        height,
        viewState
      });
      this.controllers[view.id] = this._updateController(view, viewState, viewport, oldControllers[view.id]);

      this._viewports.unshift(viewport);
    }

    for (const id in oldControllers) {
      if (oldControllers[id] && !this.controllers[id]) {
        oldControllers[id].finalize();
      }
    }

    this._buildViewportMap();
  }

  _buildViewportMap() {
    this._viewportMap = {};

    this._viewports.forEach(viewport => {
      if (viewport.id) {
        this._viewportMap[viewport.id] = this._viewportMap[viewport.id] || viewport;
      }
    });
  }

  _diffViews(newViews, oldViews) {
    if (newViews.length !== oldViews.length) {
      return true;
    }

    return newViews.some((_, i) => !newViews[i].equals(oldViews[i]));
  }

}
//# sourceMappingURL=view-manager.js.map