import { createIterable, getAccessorFromBuffer } from './iterable-utils';
import defaultTypedArrayManager from './typed-array-manager';
import assert from './assert';
import { Buffer } from '@luma.gl/core';
export default class Tesselator {
  constructor(opts = {}) {
    const {
      attributes = {}
    } = opts;
    this.typedArrayManager = defaultTypedArrayManager;
    this.indexStarts = null;
    this.vertexStarts = null;
    this.vertexCount = 0;
    this.instanceCount = 0;
    this.attributes = {};
    this._attributeDefs = attributes;
    this.opts = opts;
    this.updateGeometry(opts);
    Object.seal(this);
  }

  updateGeometry(opts) {
    Object.assign(this.opts, opts);
    const {
      data,
      buffers = {},
      getGeometry,
      geometryBuffer,
      positionFormat,
      dataChanged,
      normalize = true
    } = this.opts;
    this.data = data;
    this.getGeometry = getGeometry;
    this.positionSize = geometryBuffer && geometryBuffer.size || (positionFormat === 'XY' ? 2 : 3);
    this.buffers = buffers;
    this.normalize = normalize;

    if (geometryBuffer) {
      assert(data.startIndices, 'binary data missing startIndices');
      this.getGeometry = this.getGeometryFromBuffer(geometryBuffer);

      if (!normalize) {
        buffers.positions = geometryBuffer;
      }
    }

    this.geometryBuffer = buffers.positions;

    if (Array.isArray(dataChanged)) {
      for (const dataRange of dataChanged) {
        this._rebuildGeometry(dataRange);
      }
    } else {
      this._rebuildGeometry();
    }
  }

  updatePartialGeometry({
    startRow,
    endRow
  }) {
    this._rebuildGeometry({
      startRow,
      endRow
    });
  }

  normalizeGeometry(geometry) {
    return geometry;
  }

  updateGeometryAttributes(geometry, startIndex, size) {
    throw new Error('Not implemented');
  }

  getGeometrySize(geometry) {
    throw new Error('Not implemented');
  }

  getGeometryFromBuffer(geometryBuffer) {
    const value = geometryBuffer.value || geometryBuffer;
    assert(ArrayBuffer.isView(value), 'cannot read geometries');
    return getAccessorFromBuffer(value, {
      size: this.positionSize,
      offset: geometryBuffer.offset,
      stride: geometryBuffer.stride,
      startIndices: this.data.startIndices
    });
  }

  _allocate(instanceCount, copy) {
    const {
      attributes,
      buffers,
      _attributeDefs,
      typedArrayManager
    } = this;

    for (const name in _attributeDefs) {
      if (name in buffers) {
        typedArrayManager.release(attributes[name]);
        attributes[name] = null;
      } else {
        const def = _attributeDefs[name];
        def.copy = copy;
        attributes[name] = typedArrayManager.allocate(attributes[name], instanceCount, def);
      }
    }
  }

  _forEachGeometry(visitor, startRow, endRow) {
    const {
      data,
      getGeometry
    } = this;
    const {
      iterable,
      objectInfo
    } = createIterable(data, startRow, endRow);

    for (const object of iterable) {
      objectInfo.index++;
      const geometry = getGeometry(object, objectInfo);
      visitor(geometry, objectInfo.index);
    }
  }

  _rebuildGeometry(dataRange) {
    if (!this.data || !this.getGeometry) {
      return;
    }

    let {
      indexStarts,
      vertexStarts,
      instanceCount
    } = this;
    const {
      data,
      geometryBuffer
    } = this;
    const {
      startRow = 0,
      endRow = Infinity
    } = dataRange || {};
    const normalizedData = {};

    if (!dataRange) {
      indexStarts = [0];
      vertexStarts = [0];
    }

    if (this.normalize || !geometryBuffer) {
      this._forEachGeometry((geometry, dataIndex) => {
        geometry = this.normalizeGeometry(geometry);
        normalizedData[dataIndex] = geometry;
        vertexStarts[dataIndex + 1] = vertexStarts[dataIndex] + this.getGeometrySize(geometry);
      }, startRow, endRow);

      instanceCount = vertexStarts[vertexStarts.length - 1];
    } else if (geometryBuffer.buffer instanceof Buffer) {
      const byteStride = geometryBuffer.stride || this.positionSize * 4;
      vertexStarts = data.startIndices;
      instanceCount = vertexStarts[data.length] || geometryBuffer.buffer.byteLength / byteStride;
    } else {
      const bufferValue = geometryBuffer.value || geometryBuffer;
      const elementStride = geometryBuffer.stride / bufferValue.BYTES_PER_ELEMENT || this.positionSize;
      vertexStarts = data.startIndices;
      instanceCount = vertexStarts[data.length] || bufferValue.length / elementStride;
    }

    this._allocate(instanceCount, Boolean(dataRange));

    this.indexStarts = indexStarts;
    this.vertexStarts = vertexStarts;
    this.instanceCount = instanceCount;
    const context = {};

    this._forEachGeometry((geometry, dataIndex) => {
      geometry = normalizedData[dataIndex] || geometry;
      context.vertexStart = vertexStarts[dataIndex];
      context.indexStart = indexStarts[dataIndex];
      const vertexEnd = dataIndex < vertexStarts.length - 1 ? vertexStarts[dataIndex + 1] : instanceCount;
      context.geometrySize = vertexEnd - vertexStarts[dataIndex];
      context.geometryIndex = dataIndex;
      this.updateGeometryAttributes(geometry, context);
    }, startRow, endRow);

    this.vertexCount = indexStarts[indexStarts.length - 1];
  }

}
//# sourceMappingURL=tesselator.js.map