import log from '../utils/log';
import { createMat4, extractCameraVectors, getFrustumPlanes } from '../utils/math-utils';
import { Matrix4, Vector3, equals } from 'math.gl';
import * as mat4 from 'gl-matrix/mat4';
import { getDistanceScales, getMeterZoom, lngLatToWorld, worldToLngLat, worldToPixels, pixelsToWorld } from '@math.gl/web-mercator';
import { PROJECTION_MODE } from '../lib/constants';
const DEGREES_TO_RADIANS = Math.PI / 180;
const IDENTITY = createMat4();
const ZERO_VECTOR = [0, 0, 0];
const DEFAULT_ZOOM = 0;
const DEFAULT_DISTANCE_SCALES = {
  unitsPerMeter: [1, 1, 1],
  metersPerUnit: [1, 1, 1]
};
export default class Viewport {
  constructor(opts = {}) {
    const {
      id = null,
      x = 0,
      y = 0,
      width = 1,
      height = 1
    } = opts;
    this.id = id || this.constructor.displayName || 'viewport';
    this.x = x;
    this.y = y;
    this.width = width || 1;
    this.height = height || 1;
    this._frustumPlanes = {};

    this._initViewMatrix(opts);

    this._initProjectionMatrix(opts);

    this._initPixelMatrices();

    this.equals = this.equals.bind(this);
    this.project = this.project.bind(this);
    this.unproject = this.unproject.bind(this);
    this.projectPosition = this.projectPosition.bind(this);
    this.unprojectPosition = this.unprojectPosition.bind(this);
    this.projectFlat = this.projectFlat.bind(this);
    this.unprojectFlat = this.unprojectFlat.bind(this);
  }

  get metersPerPixel() {
    return this.distanceScales.metersPerUnit[2] / this.scale;
  }

  get projectionMode() {
    if (this.isGeospatial) {
      return this.zoom < 12 ? PROJECTION_MODE.WEB_MERCATOR : PROJECTION_MODE.WEB_MERCATOR_AUTO_OFFSET;
    }

    return PROJECTION_MODE.IDENTITY;
  }

  equals(viewport) {
    if (!(viewport instanceof Viewport)) {
      return false;
    }

    if (this === viewport) {
      return true;
    }

    return viewport.width === this.width && viewport.height === this.height && viewport.scale === this.scale && equals(viewport.projectionMatrix, this.projectionMatrix) && equals(viewport.viewMatrix, this.viewMatrix);
  }

  project(xyz, {
    topLeft = true
  } = {}) {
    const worldPosition = this.projectPosition(xyz);
    const coord = worldToPixels(worldPosition, this.pixelProjectionMatrix);
    const [x, y] = coord;
    const y2 = topLeft ? y : this.height - y;
    return xyz.length === 2 ? [x, y2] : [x, y2, coord[2]];
  }

  unproject(xyz, {
    topLeft = true,
    targetZ
  } = {}) {
    const [x, y, z] = xyz;
    const y2 = topLeft ? y : this.height - y;
    const targetZWorld = targetZ && targetZ * this.distanceScales.unitsPerMeter[2];
    const coord = pixelsToWorld([x, y2, z], this.pixelUnprojectionMatrix, targetZWorld);
    const [X, Y, Z] = this.unprojectPosition(coord);

    if (Number.isFinite(z)) {
      return [X, Y, Z];
    }

    return Number.isFinite(targetZ) ? [X, Y, targetZ] : [X, Y];
  }

  projectPosition(xyz) {
    const [X, Y] = this.projectFlat(xyz);
    const Z = (xyz[2] || 0) * this.distanceScales.unitsPerMeter[2];
    return [X, Y, Z];
  }

  unprojectPosition(xyz) {
    const [X, Y] = this.unprojectFlat(xyz);
    const Z = (xyz[2] || 0) * this.distanceScales.metersPerUnit[2];
    return [X, Y, Z];
  }

  projectFlat(xyz) {
    if (this.isGeospatial) {
      return lngLatToWorld(xyz);
    }

    return xyz;
  }

  unprojectFlat(xyz) {
    if (this.isGeospatial) {
      return worldToLngLat(xyz);
    }

    return xyz;
  }

  getBounds(options = {}) {
    const unprojectOption = {
      targetZ: options.z || 0
    };
    const topLeft = this.unproject([0, 0], unprojectOption);
    const topRight = this.unproject([this.width, 0], unprojectOption);
    const bottomLeft = this.unproject([0, this.height], unprojectOption);
    const bottomRight = this.unproject([this.width, this.height], unprojectOption);
    return [Math.min(topLeft[0], topRight[0], bottomLeft[0], bottomRight[0]), Math.min(topLeft[1], topRight[1], bottomLeft[1], bottomRight[1]), Math.max(topLeft[0], topRight[0], bottomLeft[0], bottomRight[0]), Math.max(topLeft[1], topRight[1], bottomLeft[1], bottomRight[1])];
  }

  getDistanceScales(coordinateOrigin = null) {
    if (coordinateOrigin) {
      return getDistanceScales({
        longitude: coordinateOrigin[0],
        latitude: coordinateOrigin[1],
        highPrecision: true
      });
    }

    return this.distanceScales;
  }

  containsPixel({
    x,
    y,
    width = 1,
    height = 1
  }) {
    return x < this.x + this.width && this.x < x + width && y < this.y + this.height && this.y < y + height;
  }

  getFrustumPlanes() {
    if (this._frustumPlanes.near) {
      return this._frustumPlanes;
    }

    const {
      near,
      far,
      fovyRadians,
      aspect
    } = this.projectionProps;
    Object.assign(this._frustumPlanes, getFrustumPlanes({
      aspect,
      near,
      far,
      fovyRadians,
      position: this.cameraPosition,
      direction: this.cameraDirection,
      up: this.cameraUp,
      right: this.cameraRight
    }));
    return this._frustumPlanes;
  }

  getCameraPosition() {
    return this.cameraPosition;
  }

  getCameraDirection() {
    return this.cameraDirection;
  }

  getCameraUp() {
    return this.cameraUp;
  }

  _createProjectionMatrix({
    orthographic,
    fovyRadians,
    aspect,
    focalDistance,
    near,
    far
  }) {
    return orthographic ? new Matrix4().orthographic({
      fovy: fovyRadians,
      aspect,
      focalDistance,
      near,
      far
    }) : new Matrix4().perspective({
      fovy: fovyRadians,
      aspect,
      near,
      far
    });
  }

  _initViewMatrix(opts) {
    const {
      viewMatrix = IDENTITY,
      longitude = null,
      latitude = null,
      zoom = null,
      position = null,
      modelMatrix = null,
      focalDistance = 1,
      distanceScales = null
    } = opts;
    this.isGeospatial = Number.isFinite(latitude) && Number.isFinite(longitude);
    this.zoom = zoom;

    if (!Number.isFinite(this.zoom)) {
      this.zoom = this.isGeospatial ? getMeterZoom({
        latitude
      }) + Math.log2(focalDistance) : DEFAULT_ZOOM;
    }

    const scale = Math.pow(2, this.zoom);
    this.scale = scale;
    this.distanceScales = this.isGeospatial ? getDistanceScales({
      latitude,
      longitude
    }) : distanceScales || DEFAULT_DISTANCE_SCALES;
    this.focalDistance = focalDistance;
    this.distanceScales.metersPerUnit = new Vector3(this.distanceScales.metersPerUnit);
    this.distanceScales.unitsPerMeter = new Vector3(this.distanceScales.unitsPerMeter);
    this.position = ZERO_VECTOR;
    this.meterOffset = ZERO_VECTOR;

    if (position) {
      this.position = position;
      this.modelMatrix = modelMatrix;
      this.meterOffset = modelMatrix ? modelMatrix.transformVector(position) : position;
    }

    if (this.isGeospatial) {
      this.longitude = longitude;
      this.latitude = latitude;
      this.center = this._getCenterInWorld({
        longitude,
        latitude
      });
    } else {
      this.center = position ? this.projectPosition(position) : [0, 0, 0];
    }

    this.viewMatrixUncentered = viewMatrix;
    this.viewMatrix = new Matrix4().multiplyRight(this.viewMatrixUncentered).translate(new Vector3(this.center || ZERO_VECTOR).negate());
  }

  _getCenterInWorld({
    longitude,
    latitude
  }) {
    const {
      meterOffset,
      distanceScales
    } = this;
    const center = new Vector3(this.projectPosition([longitude, latitude, 0]));

    if (meterOffset) {
      const commonPosition = new Vector3(meterOffset).scale(distanceScales.unitsPerMeter);
      center.add(commonPosition);
    }

    return center;
  }

  _initProjectionMatrix(opts) {
    const {
      projectionMatrix = null,
      orthographic = false,
      fovyRadians,
      fovy = 75,
      near = 0.1,
      far = 1000,
      focalDistance = 1
    } = opts;
    this.projectionProps = {
      orthographic,
      fovyRadians: fovyRadians || fovy * DEGREES_TO_RADIANS,
      aspect: this.width / this.height,
      focalDistance,
      near,
      far
    };
    this.projectionMatrix = projectionMatrix || this._createProjectionMatrix(this.projectionProps);
  }

  _initPixelMatrices() {
    const vpm = createMat4();
    mat4.multiply(vpm, vpm, this.projectionMatrix);
    mat4.multiply(vpm, vpm, this.viewMatrix);
    this.viewProjectionMatrix = vpm;
    this.viewMatrixInverse = mat4.invert([], this.viewMatrix) || this.viewMatrix;
    const {
      eye,
      direction,
      up,
      right
    } = extractCameraVectors({
      viewMatrix: this.viewMatrix,
      viewMatrixInverse: this.viewMatrixInverse
    });
    this.cameraPosition = eye;
    this.cameraDirection = direction;
    this.cameraUp = up;
    this.cameraRight = right;
    const viewportMatrix = createMat4();
    const pixelProjectionMatrix = createMat4();
    mat4.scale(viewportMatrix, viewportMatrix, [this.width / 2, -this.height / 2, 1]);
    mat4.translate(viewportMatrix, viewportMatrix, [1, -1, 0]);
    mat4.multiply(pixelProjectionMatrix, viewportMatrix, this.viewProjectionMatrix);
    this.pixelProjectionMatrix = pixelProjectionMatrix;
    this.viewportMatrix = viewportMatrix;
    this.pixelUnprojectionMatrix = mat4.invert(createMat4(), this.pixelProjectionMatrix);

    if (!this.pixelUnprojectionMatrix) {
      log.warn('Pixel project matrix not invertible')();
    }
  }

}
Viewport.displayName = 'Viewport';
//# sourceMappingURL=viewport.js.map