"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _classCallCheck2 = _interopRequireDefault(require("@babel/runtime/helpers/classCallCheck"));

var _createClass2 = _interopRequireDefault(require("@babel/runtime/helpers/createClass"));

var _core = require("@luma.gl/core");

var _layer = _interopRequireDefault(require("./layer"));

var _constants = require("../lifecycle/constants");

var _log = _interopRequireDefault(require("../utils/log"));

var _debug = _interopRequireDefault(require("../debug"));

var _flatten = require("../utils/flatten");

var _probe = require("probe.gl");

var _resourceManager = _interopRequireDefault(require("./resource/resource-manager"));

var _viewport = _interopRequireDefault(require("../viewports/viewport"));

var _shaderlib = require("../shaderlib");

var TRACE_SET_LAYERS = 'layerManager.setLayers';
var TRACE_ACTIVATE_VIEWPORT = 'layerManager.activateViewport';
var INITIAL_CONTEXT = Object.seal({
  layerManager: null,
  resourceManager: null,
  deck: null,
  gl: null,
  stats: null,
  shaderCache: null,
  pickingFBO: null,
  mousePosition: null,
  userData: {}
});

var layerName = function layerName(layer) {
  return layer instanceof _layer["default"] ? "".concat(layer) : !layer ? 'null' : 'invalid';
};

var LayerManager = function () {
  function LayerManager(gl) {
    var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
        deck = _ref.deck,
        stats = _ref.stats,
        viewport = _ref.viewport,
        timeline = _ref.timeline;

    (0, _classCallCheck2["default"])(this, LayerManager);
    this.lastRenderedLayers = [];
    this.layers = [];
    this.resourceManager = new _resourceManager["default"]({
      gl: gl,
      protocol: 'deck://'
    });
    this.context = Object.assign({}, INITIAL_CONTEXT, {
      layerManager: this,
      gl: gl,
      deck: deck,
      programManager: gl && (0, _shaderlib.createProgramManager)(gl),
      stats: stats || new _probe.Stats({
        id: 'deck.gl'
      }),
      viewport: viewport || new _viewport["default"]({
        id: 'DEFAULT-INITIAL-VIEWPORT'
      }),
      timeline: timeline || new _core.Timeline(),
      resourceManager: this.resourceManager
    });
    this._needsRedraw = 'Initial render';
    this._needsUpdate = false;
    this._debug = false;
    this._onError = null;
    this.activateViewport = this.activateViewport.bind(this);
    Object.seal(this);
  }

  (0, _createClass2["default"])(LayerManager, [{
    key: "finalize",
    value: function finalize() {
      this.resourceManager.finalize();
      var _iteratorNormalCompletion = true;
      var _didIteratorError = false;
      var _iteratorError = undefined;

      try {
        for (var _iterator = this.layers[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
          var layer = _step.value;

          this._finalizeLayer(layer);
        }
      } catch (err) {
        _didIteratorError = true;
        _iteratorError = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion && _iterator["return"] != null) {
            _iterator["return"]();
          }
        } finally {
          if (_didIteratorError) {
            throw _iteratorError;
          }
        }
      }
    }
  }, {
    key: "needsRedraw",
    value: function needsRedraw() {
      var opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {
        clearRedrawFlags: false
      };
      var redraw = this._needsRedraw;

      if (opts.clearRedrawFlags) {
        this._needsRedraw = false;
      }

      var _iteratorNormalCompletion2 = true;
      var _didIteratorError2 = false;
      var _iteratorError2 = undefined;

      try {
        for (var _iterator2 = this.layers[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
          var layer = _step2.value;
          var layerNeedsRedraw = layer.getNeedsRedraw(opts);
          redraw = redraw || layerNeedsRedraw;
        }
      } catch (err) {
        _didIteratorError2 = true;
        _iteratorError2 = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion2 && _iterator2["return"] != null) {
            _iterator2["return"]();
          }
        } finally {
          if (_didIteratorError2) {
            throw _iteratorError2;
          }
        }
      }

      return redraw;
    }
  }, {
    key: "needsUpdate",
    value: function needsUpdate() {
      return this._needsUpdate;
    }
  }, {
    key: "setNeedsRedraw",
    value: function setNeedsRedraw(reason) {
      this._needsRedraw = this._needsRedraw || reason;
    }
  }, {
    key: "setNeedsUpdate",
    value: function setNeedsUpdate(reason) {
      this._needsUpdate = this._needsUpdate || reason;
    }
  }, {
    key: "getLayers",
    value: function getLayers() {
      var _ref2 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
          _ref2$layerIds = _ref2.layerIds,
          layerIds = _ref2$layerIds === void 0 ? null : _ref2$layerIds;

      return layerIds ? this.layers.filter(function (layer) {
        return layerIds.find(function (layerId) {
          return layer.id.indexOf(layerId) === 0;
        });
      }) : this.layers;
    }
  }, {
    key: "setProps",
    value: function setProps(props) {
      if ('debug' in props) {
        this._debug = props.debug;
      }

      if ('userData' in props) {
        this.context.userData = props.userData;
      }

      if ('layers' in props) {
        this.setLayers(props.layers);
      }

      if ('onError' in props) {
        this._onError = props.onError;
      }
    }
  }, {
    key: "setLayers",
    value: function setLayers(newLayers) {
      var forceUpdate = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
      var shouldUpdate = forceUpdate || newLayers !== this.lastRenderedLayers;
      (0, _debug["default"])(TRACE_SET_LAYERS, this, shouldUpdate, newLayers);

      if (!shouldUpdate) {
        return this;
      }

      this.lastRenderedLayers = newLayers;
      newLayers = (0, _flatten.flatten)(newLayers, Boolean);
      var _iteratorNormalCompletion3 = true;
      var _didIteratorError3 = false;
      var _iteratorError3 = undefined;

      try {
        for (var _iterator3 = newLayers[Symbol.iterator](), _step3; !(_iteratorNormalCompletion3 = (_step3 = _iterator3.next()).done); _iteratorNormalCompletion3 = true) {
          var layer = _step3.value;
          layer.context = this.context;
        }
      } catch (err) {
        _didIteratorError3 = true;
        _iteratorError3 = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion3 && _iterator3["return"] != null) {
            _iterator3["return"]();
          }
        } finally {
          if (_didIteratorError3) {
            throw _iteratorError3;
          }
        }
      }

      this._updateLayers(this.layers, newLayers);

      return this;
    }
  }, {
    key: "updateLayers",
    value: function updateLayers() {
      var reason = this.needsUpdate();

      if (reason) {
        this.setNeedsRedraw("updating layers: ".concat(reason));
        var forceUpdate = true;
        this.setLayers(this.lastRenderedLayers, forceUpdate);
      }
    }
  }, {
    key: "activateViewport",
    value: function activateViewport(viewport) {
      (0, _debug["default"])(TRACE_ACTIVATE_VIEWPORT, this, viewport);

      if (viewport) {
        this.context.viewport = viewport;
      }

      return this;
    }
  }, {
    key: "_handleError",
    value: function _handleError(stage, error, layer) {
      if (this._onError) {
        this._onError(error, layer);
      } else {
        _log["default"].error("error during ".concat(stage, " of ").concat(layerName(layer)), error)();
      }
    }
  }, {
    key: "_updateLayers",
    value: function _updateLayers(oldLayers, newLayers) {
      var oldLayerMap = {};
      var _iteratorNormalCompletion4 = true;
      var _didIteratorError4 = false;
      var _iteratorError4 = undefined;

      try {
        for (var _iterator4 = oldLayers[Symbol.iterator](), _step4; !(_iteratorNormalCompletion4 = (_step4 = _iterator4.next()).done); _iteratorNormalCompletion4 = true) {
          var oldLayer = _step4.value;

          if (oldLayerMap[oldLayer.id]) {
            _log["default"].warn("Multiple old layers with same id ".concat(layerName(oldLayer)))();
          } else {
            oldLayerMap[oldLayer.id] = oldLayer;
          }
        }
      } catch (err) {
        _didIteratorError4 = true;
        _iteratorError4 = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion4 && _iterator4["return"] != null) {
            _iterator4["return"]();
          }
        } finally {
          if (_didIteratorError4) {
            throw _iteratorError4;
          }
        }
      }

      var generatedLayers = [];

      this._updateSublayersRecursively(newLayers, oldLayerMap, generatedLayers);

      this._finalizeOldLayers(oldLayerMap);

      var needsUpdate = false;

      for (var _i = 0, _generatedLayers = generatedLayers; _i < _generatedLayers.length; _i++) {
        var layer = _generatedLayers[_i];

        if (layer.hasUniformTransition()) {
          needsUpdate = true;
          break;
        }
      }

      this._needsUpdate = needsUpdate;
      this.layers = generatedLayers;
    }
  }, {
    key: "_updateSublayersRecursively",
    value: function _updateSublayersRecursively(newLayers, oldLayerMap, generatedLayers) {
      var _iteratorNormalCompletion5 = true;
      var _didIteratorError5 = false;
      var _iteratorError5 = undefined;

      try {
        for (var _iterator5 = newLayers[Symbol.iterator](), _step5; !(_iteratorNormalCompletion5 = (_step5 = _iterator5.next()).done); _iteratorNormalCompletion5 = true) {
          var newLayer = _step5.value;
          newLayer.context = this.context;
          var oldLayer = oldLayerMap[newLayer.id];

          if (oldLayer === null) {
            _log["default"].warn("Multiple new layers with same id ".concat(layerName(newLayer)))();
          }

          oldLayerMap[newLayer.id] = null;
          var sublayers = null;

          try {
            if (this._debug && oldLayer !== newLayer) {
              newLayer.validateProps();
            }

            if (!oldLayer) {
              this._initializeLayer(newLayer);
            } else {
              this._transferLayerState(oldLayer, newLayer);

              this._updateLayer(newLayer);
            }

            generatedLayers.push(newLayer);
            sublayers = newLayer.isComposite && newLayer.getSubLayers();
          } catch (err) {
            this._handleError('matching', err, newLayer);
          }

          if (sublayers) {
            this._updateSublayersRecursively(sublayers, oldLayerMap, generatedLayers);
          }
        }
      } catch (err) {
        _didIteratorError5 = true;
        _iteratorError5 = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion5 && _iterator5["return"] != null) {
            _iterator5["return"]();
          }
        } finally {
          if (_didIteratorError5) {
            throw _iteratorError5;
          }
        }
      }
    }
  }, {
    key: "_finalizeOldLayers",
    value: function _finalizeOldLayers(oldLayerMap) {
      for (var layerId in oldLayerMap) {
        var layer = oldLayerMap[layerId];

        if (layer) {
          this._finalizeLayer(layer);
        }
      }
    }
  }, {
    key: "_initializeLayer",
    value: function _initializeLayer(layer) {
      try {
        layer._initialize();

        layer.lifecycle = _constants.LIFECYCLE.INITIALIZED;
      } catch (err) {
        this._handleError('initialization', err, layer);
      }
    }
  }, {
    key: "_transferLayerState",
    value: function _transferLayerState(oldLayer, newLayer) {
      newLayer._transferState(oldLayer);

      newLayer.lifecycle = _constants.LIFECYCLE.MATCHED;

      if (newLayer !== oldLayer) {
        oldLayer.lifecycle = _constants.LIFECYCLE.AWAITING_GC;
      }
    }
  }, {
    key: "_updateLayer",
    value: function _updateLayer(layer) {
      try {
        layer._update();
      } catch (err) {
        this._handleError('update', err, layer);
      }
    }
  }, {
    key: "_finalizeLayer",
    value: function _finalizeLayer(layer) {
      this._needsRedraw = this._needsRedraw || "finalized ".concat(layerName(layer));
      layer.lifecycle = _constants.LIFECYCLE.AWAITING_FINALIZATION;

      try {
        layer._finalize();

        layer.lifecycle = _constants.LIFECYCLE.FINALIZED;
      } catch (err) {
        this._handleError('finalization', err, layer);
      }
    }
  }]);
  return LayerManager;
}();

exports["default"] = LayerManager;
//# sourceMappingURL=layer-manager.js.map