"use strict";

var _interopRequireWildcard = require("@babel/runtime/helpers/interopRequireWildcard");

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _slicedToArray2 = _interopRequireDefault(require("@babel/runtime/helpers/slicedToArray"));

var _classCallCheck2 = _interopRequireDefault(require("@babel/runtime/helpers/classCallCheck"));

var _createClass2 = _interopRequireDefault(require("@babel/runtime/helpers/createClass"));

var _log = _interopRequireDefault(require("../utils/log"));

var _mathUtils = require("../utils/math-utils");

var _math = require("math.gl");

var mat4 = _interopRequireWildcard(require("gl-matrix/mat4"));

var _webMercator = require("@math.gl/web-mercator");

var _constants = require("../lib/constants");

var DEGREES_TO_RADIANS = Math.PI / 180;
var IDENTITY = (0, _mathUtils.createMat4)();
var ZERO_VECTOR = [0, 0, 0];
var DEFAULT_ZOOM = 0;
var DEFAULT_DISTANCE_SCALES = {
  unitsPerMeter: [1, 1, 1],
  metersPerUnit: [1, 1, 1]
};

var Viewport = function () {
  function Viewport() {
    var opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    (0, _classCallCheck2["default"])(this, Viewport);
    var _opts$id = opts.id,
        id = _opts$id === void 0 ? null : _opts$id,
        _opts$x = opts.x,
        x = _opts$x === void 0 ? 0 : _opts$x,
        _opts$y = opts.y,
        y = _opts$y === void 0 ? 0 : _opts$y,
        _opts$width = opts.width,
        width = _opts$width === void 0 ? 1 : _opts$width,
        _opts$height = opts.height,
        height = _opts$height === void 0 ? 1 : _opts$height;
    this.id = id || this.constructor.displayName || 'viewport';
    this.x = x;
    this.y = y;
    this.width = width || 1;
    this.height = height || 1;
    this._frustumPlanes = {};

    this._initViewMatrix(opts);

    this._initProjectionMatrix(opts);

    this._initPixelMatrices();

    this.equals = this.equals.bind(this);
    this.project = this.project.bind(this);
    this.unproject = this.unproject.bind(this);
    this.projectPosition = this.projectPosition.bind(this);
    this.unprojectPosition = this.unprojectPosition.bind(this);
    this.projectFlat = this.projectFlat.bind(this);
    this.unprojectFlat = this.unprojectFlat.bind(this);
  }

  (0, _createClass2["default"])(Viewport, [{
    key: "equals",
    value: function equals(viewport) {
      if (!(viewport instanceof Viewport)) {
        return false;
      }

      if (this === viewport) {
        return true;
      }

      return viewport.width === this.width && viewport.height === this.height && viewport.scale === this.scale && (0, _math.equals)(viewport.projectionMatrix, this.projectionMatrix) && (0, _math.equals)(viewport.viewMatrix, this.viewMatrix);
    }
  }, {
    key: "project",
    value: function project(xyz) {
      var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
          _ref$topLeft = _ref.topLeft,
          topLeft = _ref$topLeft === void 0 ? true : _ref$topLeft;

      var worldPosition = this.projectPosition(xyz);
      var coord = (0, _webMercator.worldToPixels)(worldPosition, this.pixelProjectionMatrix);

      var _coord = (0, _slicedToArray2["default"])(coord, 2),
          x = _coord[0],
          y = _coord[1];

      var y2 = topLeft ? y : this.height - y;
      return xyz.length === 2 ? [x, y2] : [x, y2, coord[2]];
    }
  }, {
    key: "unproject",
    value: function unproject(xyz) {
      var _ref2 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {},
          _ref2$topLeft = _ref2.topLeft,
          topLeft = _ref2$topLeft === void 0 ? true : _ref2$topLeft,
          targetZ = _ref2.targetZ;

      var _xyz = (0, _slicedToArray2["default"])(xyz, 3),
          x = _xyz[0],
          y = _xyz[1],
          z = _xyz[2];

      var y2 = topLeft ? y : this.height - y;
      var targetZWorld = targetZ && targetZ * this.distanceScales.unitsPerMeter[2];
      var coord = (0, _webMercator.pixelsToWorld)([x, y2, z], this.pixelUnprojectionMatrix, targetZWorld);

      var _this$unprojectPositi = this.unprojectPosition(coord),
          _this$unprojectPositi2 = (0, _slicedToArray2["default"])(_this$unprojectPositi, 3),
          X = _this$unprojectPositi2[0],
          Y = _this$unprojectPositi2[1],
          Z = _this$unprojectPositi2[2];

      if (Number.isFinite(z)) {
        return [X, Y, Z];
      }

      return Number.isFinite(targetZ) ? [X, Y, targetZ] : [X, Y];
    }
  }, {
    key: "projectPosition",
    value: function projectPosition(xyz) {
      var _this$projectFlat = this.projectFlat(xyz),
          _this$projectFlat2 = (0, _slicedToArray2["default"])(_this$projectFlat, 2),
          X = _this$projectFlat2[0],
          Y = _this$projectFlat2[1];

      var Z = (xyz[2] || 0) * this.distanceScales.unitsPerMeter[2];
      return [X, Y, Z];
    }
  }, {
    key: "unprojectPosition",
    value: function unprojectPosition(xyz) {
      var _this$unprojectFlat = this.unprojectFlat(xyz),
          _this$unprojectFlat2 = (0, _slicedToArray2["default"])(_this$unprojectFlat, 2),
          X = _this$unprojectFlat2[0],
          Y = _this$unprojectFlat2[1];

      var Z = (xyz[2] || 0) * this.distanceScales.metersPerUnit[2];
      return [X, Y, Z];
    }
  }, {
    key: "projectFlat",
    value: function projectFlat(xyz) {
      if (this.isGeospatial) {
        return (0, _webMercator.lngLatToWorld)(xyz);
      }

      return xyz;
    }
  }, {
    key: "unprojectFlat",
    value: function unprojectFlat(xyz) {
      if (this.isGeospatial) {
        return (0, _webMercator.worldToLngLat)(xyz);
      }

      return xyz;
    }
  }, {
    key: "getBounds",
    value: function getBounds() {
      var options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      var unprojectOption = {
        targetZ: options.z || 0
      };
      var topLeft = this.unproject([0, 0], unprojectOption);
      var topRight = this.unproject([this.width, 0], unprojectOption);
      var bottomLeft = this.unproject([0, this.height], unprojectOption);
      var bottomRight = this.unproject([this.width, this.height], unprojectOption);
      return [Math.min(topLeft[0], topRight[0], bottomLeft[0], bottomRight[0]), Math.min(topLeft[1], topRight[1], bottomLeft[1], bottomRight[1]), Math.max(topLeft[0], topRight[0], bottomLeft[0], bottomRight[0]), Math.max(topLeft[1], topRight[1], bottomLeft[1], bottomRight[1])];
    }
  }, {
    key: "getDistanceScales",
    value: function getDistanceScales() {
      var coordinateOrigin = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : null;

      if (coordinateOrigin) {
        return (0, _webMercator.getDistanceScales)({
          longitude: coordinateOrigin[0],
          latitude: coordinateOrigin[1],
          highPrecision: true
        });
      }

      return this.distanceScales;
    }
  }, {
    key: "containsPixel",
    value: function containsPixel(_ref3) {
      var x = _ref3.x,
          y = _ref3.y,
          _ref3$width = _ref3.width,
          width = _ref3$width === void 0 ? 1 : _ref3$width,
          _ref3$height = _ref3.height,
          height = _ref3$height === void 0 ? 1 : _ref3$height;
      return x < this.x + this.width && this.x < x + width && y < this.y + this.height && this.y < y + height;
    }
  }, {
    key: "getFrustumPlanes",
    value: function getFrustumPlanes() {
      if (this._frustumPlanes.near) {
        return this._frustumPlanes;
      }

      var _this$projectionProps = this.projectionProps,
          near = _this$projectionProps.near,
          far = _this$projectionProps.far,
          fovyRadians = _this$projectionProps.fovyRadians,
          aspect = _this$projectionProps.aspect;
      Object.assign(this._frustumPlanes, (0, _mathUtils.getFrustumPlanes)({
        aspect: aspect,
        near: near,
        far: far,
        fovyRadians: fovyRadians,
        position: this.cameraPosition,
        direction: this.cameraDirection,
        up: this.cameraUp,
        right: this.cameraRight
      }));
      return this._frustumPlanes;
    }
  }, {
    key: "getCameraPosition",
    value: function getCameraPosition() {
      return this.cameraPosition;
    }
  }, {
    key: "getCameraDirection",
    value: function getCameraDirection() {
      return this.cameraDirection;
    }
  }, {
    key: "getCameraUp",
    value: function getCameraUp() {
      return this.cameraUp;
    }
  }, {
    key: "_createProjectionMatrix",
    value: function _createProjectionMatrix(_ref4) {
      var orthographic = _ref4.orthographic,
          fovyRadians = _ref4.fovyRadians,
          aspect = _ref4.aspect,
          focalDistance = _ref4.focalDistance,
          near = _ref4.near,
          far = _ref4.far;
      return orthographic ? new _math.Matrix4().orthographic({
        fovy: fovyRadians,
        aspect: aspect,
        focalDistance: focalDistance,
        near: near,
        far: far
      }) : new _math.Matrix4().perspective({
        fovy: fovyRadians,
        aspect: aspect,
        near: near,
        far: far
      });
    }
  }, {
    key: "_initViewMatrix",
    value: function _initViewMatrix(opts) {
      var _opts$viewMatrix = opts.viewMatrix,
          viewMatrix = _opts$viewMatrix === void 0 ? IDENTITY : _opts$viewMatrix,
          _opts$longitude = opts.longitude,
          longitude = _opts$longitude === void 0 ? null : _opts$longitude,
          _opts$latitude = opts.latitude,
          latitude = _opts$latitude === void 0 ? null : _opts$latitude,
          _opts$zoom = opts.zoom,
          zoom = _opts$zoom === void 0 ? null : _opts$zoom,
          _opts$position = opts.position,
          position = _opts$position === void 0 ? null : _opts$position,
          _opts$modelMatrix = opts.modelMatrix,
          modelMatrix = _opts$modelMatrix === void 0 ? null : _opts$modelMatrix,
          _opts$focalDistance = opts.focalDistance,
          focalDistance = _opts$focalDistance === void 0 ? 1 : _opts$focalDistance,
          _opts$distanceScales = opts.distanceScales,
          distanceScales = _opts$distanceScales === void 0 ? null : _opts$distanceScales;
      this.isGeospatial = Number.isFinite(latitude) && Number.isFinite(longitude);
      this.zoom = zoom;

      if (!Number.isFinite(this.zoom)) {
        this.zoom = this.isGeospatial ? (0, _webMercator.getMeterZoom)({
          latitude: latitude
        }) + Math.log2(focalDistance) : DEFAULT_ZOOM;
      }

      var scale = Math.pow(2, this.zoom);
      this.scale = scale;
      this.distanceScales = this.isGeospatial ? (0, _webMercator.getDistanceScales)({
        latitude: latitude,
        longitude: longitude
      }) : distanceScales || DEFAULT_DISTANCE_SCALES;
      this.focalDistance = focalDistance;
      this.distanceScales.metersPerUnit = new _math.Vector3(this.distanceScales.metersPerUnit);
      this.distanceScales.unitsPerMeter = new _math.Vector3(this.distanceScales.unitsPerMeter);
      this.position = ZERO_VECTOR;
      this.meterOffset = ZERO_VECTOR;

      if (position) {
        this.position = position;
        this.modelMatrix = modelMatrix;
        this.meterOffset = modelMatrix ? modelMatrix.transformVector(position) : position;
      }

      if (this.isGeospatial) {
        this.longitude = longitude;
        this.latitude = latitude;
        this.center = this._getCenterInWorld({
          longitude: longitude,
          latitude: latitude
        });
      } else {
        this.center = position ? this.projectPosition(position) : [0, 0, 0];
      }

      this.viewMatrixUncentered = viewMatrix;
      this.viewMatrix = new _math.Matrix4().multiplyRight(this.viewMatrixUncentered).translate(new _math.Vector3(this.center || ZERO_VECTOR).negate());
    }
  }, {
    key: "_getCenterInWorld",
    value: function _getCenterInWorld(_ref5) {
      var longitude = _ref5.longitude,
          latitude = _ref5.latitude;
      var meterOffset = this.meterOffset,
          distanceScales = this.distanceScales;
      var center = new _math.Vector3(this.projectPosition([longitude, latitude, 0]));

      if (meterOffset) {
        var commonPosition = new _math.Vector3(meterOffset).scale(distanceScales.unitsPerMeter);
        center.add(commonPosition);
      }

      return center;
    }
  }, {
    key: "_initProjectionMatrix",
    value: function _initProjectionMatrix(opts) {
      var _opts$projectionMatri = opts.projectionMatrix,
          projectionMatrix = _opts$projectionMatri === void 0 ? null : _opts$projectionMatri,
          _opts$orthographic = opts.orthographic,
          orthographic = _opts$orthographic === void 0 ? false : _opts$orthographic,
          fovyRadians = opts.fovyRadians,
          _opts$fovy = opts.fovy,
          fovy = _opts$fovy === void 0 ? 75 : _opts$fovy,
          _opts$near = opts.near,
          near = _opts$near === void 0 ? 0.1 : _opts$near,
          _opts$far = opts.far,
          far = _opts$far === void 0 ? 1000 : _opts$far,
          _opts$focalDistance2 = opts.focalDistance,
          focalDistance = _opts$focalDistance2 === void 0 ? 1 : _opts$focalDistance2;
      this.projectionProps = {
        orthographic: orthographic,
        fovyRadians: fovyRadians || fovy * DEGREES_TO_RADIANS,
        aspect: this.width / this.height,
        focalDistance: focalDistance,
        near: near,
        far: far
      };
      this.projectionMatrix = projectionMatrix || this._createProjectionMatrix(this.projectionProps);
    }
  }, {
    key: "_initPixelMatrices",
    value: function _initPixelMatrices() {
      var vpm = (0, _mathUtils.createMat4)();
      mat4.multiply(vpm, vpm, this.projectionMatrix);
      mat4.multiply(vpm, vpm, this.viewMatrix);
      this.viewProjectionMatrix = vpm;
      this.viewMatrixInverse = mat4.invert([], this.viewMatrix) || this.viewMatrix;

      var _extractCameraVectors = (0, _mathUtils.extractCameraVectors)({
        viewMatrix: this.viewMatrix,
        viewMatrixInverse: this.viewMatrixInverse
      }),
          eye = _extractCameraVectors.eye,
          direction = _extractCameraVectors.direction,
          up = _extractCameraVectors.up,
          right = _extractCameraVectors.right;

      this.cameraPosition = eye;
      this.cameraDirection = direction;
      this.cameraUp = up;
      this.cameraRight = right;
      var viewportMatrix = (0, _mathUtils.createMat4)();
      var pixelProjectionMatrix = (0, _mathUtils.createMat4)();
      mat4.scale(viewportMatrix, viewportMatrix, [this.width / 2, -this.height / 2, 1]);
      mat4.translate(viewportMatrix, viewportMatrix, [1, -1, 0]);
      mat4.multiply(pixelProjectionMatrix, viewportMatrix, this.viewProjectionMatrix);
      this.pixelProjectionMatrix = pixelProjectionMatrix;
      this.viewportMatrix = viewportMatrix;
      this.pixelUnprojectionMatrix = mat4.invert((0, _mathUtils.createMat4)(), this.pixelProjectionMatrix);

      if (!this.pixelUnprojectionMatrix) {
        _log["default"].warn('Pixel project matrix not invertible')();
      }
    }
  }, {
    key: "metersPerPixel",
    get: function get() {
      return this.distanceScales.metersPerUnit[2] / this.scale;
    }
  }, {
    key: "projectionMode",
    get: function get() {
      if (this.isGeospatial) {
        return this.zoom < 12 ? _constants.PROJECTION_MODE.WEB_MERCATOR : _constants.PROJECTION_MODE.WEB_MERCATOR_AUTO_OFFSET;
      }

      return _constants.PROJECTION_MODE.IDENTITY;
    }
  }]);
  return Viewport;
}();

exports["default"] = Viewport;
Viewport.displayName = 'Viewport';
//# sourceMappingURL=viewport.js.map