import Effect from '../lib/effect';
import ScreenPass from '../passes/screen-pass';
import { normalizeShaderModule } from '@luma.gl/core';
export default class PostProcessEffect extends Effect {
  constructor(module, props = {}) {
    super(props);
    this.id = "".concat(module.name, "-pass");
    normalizeShaderModule(module);
    this.module = module;
  }

  postRender(gl, params) {
    if (!this.passes) {
      this.passes = createPasses(gl, this.module, this.id, this.props);
    }

    const {
      target
    } = params;
    let inputBuffer = params.inputBuffer;
    let outputBuffer = params.swapBuffer;

    for (let index = 0; index < this.passes.length; index++) {
      if (target && index === this.passes.length - 1) {
        outputBuffer = target;
      }

      this.passes[index].render({
        inputBuffer,
        outputBuffer
      });
      const switchBuffer = outputBuffer;
      outputBuffer = inputBuffer;
      inputBuffer = switchBuffer;
    }

    return inputBuffer;
  }

  cleanup() {
    if (this.passes) {
      for (const pass of this.passes) {
        pass.delete();
      }

      this.passes = null;
    }
  }

}

function createPasses(gl, module, id, moduleSettings) {
  if (module.filter || module.sampler) {
    const fs = getFragmentShaderForRenderPass(module);
    const pass = new ScreenPass(gl, {
      id,
      module,
      fs,
      moduleSettings
    });
    return [pass];
  }

  const passes = module.passes || [];
  return passes.map((pass, index) => {
    const fs = getFragmentShaderForRenderPass(module, pass);
    const idn = "".concat(id, "-").concat(index);
    return new ScreenPass(gl, {
      id: idn,
      module,
      fs,
      moduleSettings
    });
  });
}

const FILTER_FS_TEMPLATE = func => "uniform sampler2D texture;\nuniform vec2 texSize;\n\nvarying vec2 position;\nvarying vec2 coordinate;\nvarying vec2 uv;\n\nvoid main() {\n  vec2 texCoord = coordinate;\n\n  gl_FragColor = texture2D(texture, texCoord);\n  gl_FragColor = ".concat(func, "(gl_FragColor, texSize, texCoord);\n}\n");

const SAMPLER_FS_TEMPLATE = func => "uniform sampler2D texture;\nuniform vec2 texSize;\n\nvarying vec2 position;\nvarying vec2 coordinate;\nvarying vec2 uv;\n\nvoid main() {\n  vec2 texCoord = coordinate;\n\n  gl_FragColor = ".concat(func, "(texture, texSize, texCoord);\n}\n");

function getFragmentShaderForRenderPass(module, pass = module) {
  if (pass.filter) {
    const func = typeof pass.filter === 'string' ? pass.filter : "".concat(module.name, "_filterColor");
    return FILTER_FS_TEMPLATE(func);
  }

  if (pass.sampler) {
    const func = typeof pass.sampler === 'string' ? pass.sampler : "".concat(module.name, "_sampleColor");
    return SAMPLER_FS_TEMPLATE(func);
  }

  return null;
}
//# sourceMappingURL=post-process-effect.js.map