import { Framebuffer, Texture2D, isWebGL2, readPixelsToArray, cssToDeviceRatio, cssToDevicePixels } from '@luma.gl/core';
import assert from '../utils/assert';
import log from '../utils/log';
import PickLayersPass from '../passes/pick-layers-pass';
import { getClosestObject, getUniqueObjects } from './picking/query-object';
import { processPickInfo, getLayerPickingInfo } from './picking/pick-info';
export default class DeckPicker {
  constructor(gl) {
    this.gl = gl;
    this.pickingFBO = null;
    this.pickLayersPass = new PickLayersPass(gl);
    this.layerFilter = null;
    this.lastPickedInfo = {
      index: -1,
      layerId: null,
      info: null
    };
    this._onError = null;
  }

  setProps(props) {
    if ('layerFilter' in props) {
      this.layerFilter = props.layerFilter;
    }

    if ('onError' in props) {
      this._onError = props.onError;
    }
  }

  finalize() {
    if (this.pickingFBO) {
      this.pickingFBO.delete();
    }

    if (this.depthFBO) {
      this.depthFBO.color.delete();
      this.depthFBO.delete();
    }
  }

  pickObject(opts) {
    return this._pickClosestObject(opts);
  }

  pickObjects(opts) {
    return this._pickVisibleObjects(opts);
  }

  getLastPickedObject({
    x,
    y,
    layers,
    viewports
  }, lastPickedInfo = this.lastPickedInfo.info) {
    const lastPickedLayerId = lastPickedInfo && lastPickedInfo.layer && lastPickedInfo.layer.id;
    const layer = lastPickedLayerId ? layers.find(l => l.id === lastPickedLayerId) : null;
    const coordinate = viewports[0] && viewports[0].unproject([x, y]);
    const info = {
      x,
      y,
      coordinate,
      lngLat: coordinate,
      layer
    };

    if (layer) {
      return Object.assign({}, lastPickedInfo, info);
    }

    return Object.assign(info, {
      color: null,
      object: null,
      index: -1
    });
  }

  _resizeBuffer() {
    const {
      gl
    } = this;

    if (!this.pickingFBO) {
      this.pickingFBO = new Framebuffer(gl);

      if (Framebuffer.isSupported(gl, {
        colorBufferFloat: true
      })) {
        this.depthFBO = new Framebuffer(gl);
        this.depthFBO.attach({
          [36064]: new Texture2D(gl, {
            format: isWebGL2(gl) ? 34836 : 6408,
            type: 5126
          })
        });
      }
    }

    this.pickingFBO.resize({
      width: gl.canvas.width,
      height: gl.canvas.height
    });

    if (this.depthFBO) {
      this.depthFBO.resize({
        width: gl.canvas.width,
        height: gl.canvas.height
      });
    }

    return this.pickingFBO;
  }

  _getPickable(layers) {
    const pickableLayers = layers.filter(layer => layer.isPickable() && !layer.isComposite);

    if (pickableLayers.length > 255) {
      log.warn('Too many pickable layers, only picking the first 255')();
      return pickableLayers.slice(0, 255);
    }

    return pickableLayers;
  }

  _pickClosestObject({
    layers,
    viewports,
    x,
    y,
    radius = 0,
    depth = 1,
    mode = 'query',
    unproject3D,
    onViewportActive
  }) {
    layers = this._getPickable(layers);

    this._resizeBuffer();

    const pixelRatio = cssToDeviceRatio(this.gl);
    const devicePixelRange = cssToDevicePixels(this.gl, [x, y], true);
    const devicePixel = [devicePixelRange.x + Math.floor(devicePixelRange.width / 2), devicePixelRange.y + Math.floor(devicePixelRange.height / 2)];
    const deviceRadius = Math.round(radius * pixelRatio);
    const {
      width,
      height
    } = this.pickingFBO;

    const deviceRect = this._getPickingRect({
      deviceX: devicePixel[0],
      deviceY: devicePixel[1],
      deviceRadius,
      deviceWidth: width,
      deviceHeight: height
    });

    let infos;
    const result = [];
    const affectedLayers = {};

    for (let i = 0; i < depth; i++) {
      const pickedColors = deviceRect && this._drawAndSample({
        layers,
        viewports,
        onViewportActive,
        deviceRect,
        pass: "picking:".concat(mode),
        redrawReason: mode
      });

      const pickInfo = getClosestObject({
        pickedColors,
        layers,
        deviceX: devicePixel[0],
        deviceY: devicePixel[1],
        deviceRadius,
        deviceRect
      });
      let z;

      if (pickInfo.pickedLayer && unproject3D && this.depthFBO) {
        const zValues = this._drawAndSample({
          layers: [pickInfo.pickedLayer],
          viewports,
          onViewportActive,
          deviceRect: {
            x: pickInfo.pickedX,
            y: pickInfo.pickedY,
            width: 1,
            height: 1
          },
          pass: "picking:".concat(mode),
          redrawReason: 'pick-z',
          pickZ: true
        });

        z = zValues[0] * viewports[0].distanceScales.metersPerUnit[2] + viewports[0].position[2];
      }

      if (pickInfo.pickedColor && i + 1 < depth) {
        const layerId = pickInfo.pickedColor[3] - 1;
        affectedLayers[layerId] = true;
        layers[layerId].clearPickingColor(pickInfo.pickedColor);
      }

      infos = processPickInfo({
        pickInfo,
        lastPickedInfo: this.lastPickedInfo,
        mode,
        layers,
        viewports,
        x,
        y,
        z,
        pixelRatio
      });

      for (const info of infos.values()) {
        if (info.layer) {
          result.push(info);
        }
      }

      if (!pickInfo.pickedColor) {
        break;
      }
    }

    for (const layerId in affectedLayers) {
      layers[layerId].restorePickingColors();
    }

    return {
      result,
      emptyInfo: infos && infos.get(null)
    };
  }

  _pickVisibleObjects({
    layers,
    viewports,
    x,
    y,
    width = 1,
    height = 1,
    mode = 'query',
    onViewportActive
  }) {
    layers = this._getPickable(layers);

    this._resizeBuffer();

    const pixelRatio = cssToDeviceRatio(this.gl);
    const leftTop = cssToDevicePixels(this.gl, [x, y], true);
    const deviceLeft = leftTop.x;
    const deviceTop = leftTop.y + leftTop.height;
    const rightBottom = cssToDevicePixels(this.gl, [x + width, y + height], true);
    const deviceRight = rightBottom.x + rightBottom.width;
    const deviceBottom = rightBottom.y;
    const deviceRect = {
      x: deviceLeft,
      y: deviceBottom,
      width: deviceRight - deviceLeft,
      height: deviceTop - deviceBottom
    };

    const pickedColors = this._drawAndSample({
      layers,
      viewports,
      onViewportActive,
      deviceRect,
      pass: "picking:".concat(mode),
      redrawReason: mode
    });

    const pickInfos = getUniqueObjects({
      pickedColors,
      layers
    });
    const uniqueInfos = new Map();
    pickInfos.forEach(pickInfo => {
      let info = {
        color: pickInfo.pickedColor,
        layer: null,
        index: pickInfo.pickedObjectIndex,
        picked: true,
        x,
        y,
        width,
        height,
        pixelRatio
      };
      info = getLayerPickingInfo({
        layer: pickInfo.pickedLayer,
        info,
        mode
      });

      if (!uniqueInfos.has(info.object)) {
        uniqueInfos.set(info.object, info);
      }
    });
    return Array.from(uniqueInfos.values());
  }

  _drawAndSample({
    layers,
    viewports,
    onViewportActive,
    deviceRect,
    pass,
    redrawReason,
    pickZ
  }) {
    assert(deviceRect.width > 0 && deviceRect.height > 0);

    if (layers.length < 1) {
      return null;
    }

    const pickingFBO = pickZ ? this.depthFBO : this.pickingFBO;
    this.pickLayersPass.render({
      layers,
      layerFilter: this.layerFilter,
      onError: this._onError,
      viewports,
      onViewportActive,
      pickingFBO,
      deviceRect,
      pass,
      redrawReason,
      pickZ
    });
    const {
      x,
      y,
      width,
      height
    } = deviceRect;
    const pickedColors = new (pickZ ? Float32Array : Uint8Array)(width * height * 4);
    readPixelsToArray(pickingFBO, {
      sourceX: x,
      sourceY: y,
      sourceWidth: width,
      sourceHeight: height,
      target: pickedColors
    });
    return pickedColors;
  }

  _getPickingRect({
    deviceX,
    deviceY,
    deviceRadius,
    deviceWidth,
    deviceHeight
  }) {
    const x = Math.max(0, deviceX - deviceRadius);
    const y = Math.max(0, deviceY - deviceRadius);
    const width = Math.min(deviceWidth, deviceX + deviceRadius + 1) - x;
    const height = Math.min(deviceHeight, deviceY + deviceRadius + 1) - y;

    if (width <= 0 || height <= 0) {
      return null;
    }

    return {
      x,
      y,
      width,
      height
    };
  }

}
//# sourceMappingURL=deck-picker.js.map