import { Timeline } from '@luma.gl/core';
import Layer from './layer';
import { LIFECYCLE } from '../lifecycle/constants';
import log from '../utils/log';
import debug from '../debug';
import { flatten } from '../utils/flatten';
import { Stats } from 'probe.gl';
import ResourceManager from './resource/resource-manager';
import Viewport from '../viewports/viewport';
import { createProgramManager } from '../shaderlib';
const TRACE_SET_LAYERS = 'layerManager.setLayers';
const TRACE_ACTIVATE_VIEWPORT = 'layerManager.activateViewport';
const INITIAL_CONTEXT = Object.seal({
  layerManager: null,
  resourceManager: null,
  deck: null,
  gl: null,
  stats: null,
  shaderCache: null,
  pickingFBO: null,
  mousePosition: null,
  userData: {}
});

const layerName = layer => layer instanceof Layer ? "".concat(layer) : !layer ? 'null' : 'invalid';

export default class LayerManager {
  constructor(gl, {
    deck,
    stats,
    viewport,
    timeline
  } = {}) {
    this.lastRenderedLayers = [];
    this.layers = [];
    this.resourceManager = new ResourceManager({
      gl,
      protocol: 'deck://'
    });
    this.context = Object.assign({}, INITIAL_CONTEXT, {
      layerManager: this,
      gl,
      deck,
      programManager: gl && createProgramManager(gl),
      stats: stats || new Stats({
        id: 'deck.gl'
      }),
      viewport: viewport || new Viewport({
        id: 'DEFAULT-INITIAL-VIEWPORT'
      }),
      timeline: timeline || new Timeline(),
      resourceManager: this.resourceManager
    });
    this._needsRedraw = 'Initial render';
    this._needsUpdate = false;
    this._debug = false;
    this._onError = null;
    this.activateViewport = this.activateViewport.bind(this);
    Object.seal(this);
  }

  finalize() {
    this.resourceManager.finalize();

    for (const layer of this.layers) {
      this._finalizeLayer(layer);
    }
  }

  needsRedraw(opts = {
    clearRedrawFlags: false
  }) {
    let redraw = this._needsRedraw;

    if (opts.clearRedrawFlags) {
      this._needsRedraw = false;
    }

    for (const layer of this.layers) {
      const layerNeedsRedraw = layer.getNeedsRedraw(opts);
      redraw = redraw || layerNeedsRedraw;
    }

    return redraw;
  }

  needsUpdate() {
    return this._needsUpdate;
  }

  setNeedsRedraw(reason) {
    this._needsRedraw = this._needsRedraw || reason;
  }

  setNeedsUpdate(reason) {
    this._needsUpdate = this._needsUpdate || reason;
  }

  getLayers({
    layerIds = null
  } = {}) {
    return layerIds ? this.layers.filter(layer => layerIds.find(layerId => layer.id.indexOf(layerId) === 0)) : this.layers;
  }

  setProps(props) {
    if ('debug' in props) {
      this._debug = props.debug;
    }

    if ('userData' in props) {
      this.context.userData = props.userData;
    }

    if ('layers' in props) {
      this.setLayers(props.layers);
    }

    if ('onError' in props) {
      this._onError = props.onError;
    }
  }

  setLayers(newLayers, forceUpdate = false) {
    const shouldUpdate = forceUpdate || newLayers !== this.lastRenderedLayers;
    debug(TRACE_SET_LAYERS, this, shouldUpdate, newLayers);

    if (!shouldUpdate) {
      return this;
    }

    this.lastRenderedLayers = newLayers;
    newLayers = flatten(newLayers, Boolean);

    for (const layer of newLayers) {
      layer.context = this.context;
    }

    this._updateLayers(this.layers, newLayers);

    return this;
  }

  updateLayers() {
    const reason = this.needsUpdate();

    if (reason) {
      this.setNeedsRedraw("updating layers: ".concat(reason));
      const forceUpdate = true;
      this.setLayers(this.lastRenderedLayers, forceUpdate);
    }
  }

  activateViewport(viewport) {
    debug(TRACE_ACTIVATE_VIEWPORT, this, viewport);

    if (viewport) {
      this.context.viewport = viewport;
    }

    return this;
  }

  _handleError(stage, error, layer) {
    if (this._onError) {
      this._onError(error, layer);
    } else {
      log.error("error during ".concat(stage, " of ").concat(layerName(layer)), error)();
    }
  }

  _updateLayers(oldLayers, newLayers) {
    const oldLayerMap = {};

    for (const oldLayer of oldLayers) {
      if (oldLayerMap[oldLayer.id]) {
        log.warn("Multiple old layers with same id ".concat(layerName(oldLayer)))();
      } else {
        oldLayerMap[oldLayer.id] = oldLayer;
      }
    }

    const generatedLayers = [];

    this._updateSublayersRecursively(newLayers, oldLayerMap, generatedLayers);

    this._finalizeOldLayers(oldLayerMap);

    let needsUpdate = false;

    for (const layer of generatedLayers) {
      if (layer.hasUniformTransition()) {
        needsUpdate = true;
        break;
      }
    }

    this._needsUpdate = needsUpdate;
    this.layers = generatedLayers;
  }

  _updateSublayersRecursively(newLayers, oldLayerMap, generatedLayers) {
    for (const newLayer of newLayers) {
      newLayer.context = this.context;
      const oldLayer = oldLayerMap[newLayer.id];

      if (oldLayer === null) {
        log.warn("Multiple new layers with same id ".concat(layerName(newLayer)))();
      }

      oldLayerMap[newLayer.id] = null;
      let sublayers = null;

      try {
        if (this._debug && oldLayer !== newLayer) {
          newLayer.validateProps();
        }

        if (!oldLayer) {
          this._initializeLayer(newLayer);
        } else {
          this._transferLayerState(oldLayer, newLayer);

          this._updateLayer(newLayer);
        }

        generatedLayers.push(newLayer);
        sublayers = newLayer.isComposite && newLayer.getSubLayers();
      } catch (err) {
        this._handleError('matching', err, newLayer);
      }

      if (sublayers) {
        this._updateSublayersRecursively(sublayers, oldLayerMap, generatedLayers);
      }
    }
  }

  _finalizeOldLayers(oldLayerMap) {
    for (const layerId in oldLayerMap) {
      const layer = oldLayerMap[layerId];

      if (layer) {
        this._finalizeLayer(layer);
      }
    }
  }

  _initializeLayer(layer) {
    try {
      layer._initialize();

      layer.lifecycle = LIFECYCLE.INITIALIZED;
    } catch (err) {
      this._handleError('initialization', err, layer);
    }
  }

  _transferLayerState(oldLayer, newLayer) {
    newLayer._transferState(oldLayer);

    newLayer.lifecycle = LIFECYCLE.MATCHED;

    if (newLayer !== oldLayer) {
      oldLayer.lifecycle = LIFECYCLE.AWAITING_GC;
    }
  }

  _updateLayer(layer) {
    try {
      layer._update();
    } catch (err) {
      this._handleError('update', err, layer);
    }
  }

  _finalizeLayer(layer) {
    this._needsRedraw = this._needsRedraw || "finalized ".concat(layerName(layer));
    layer.lifecycle = LIFECYCLE.AWAITING_FINALIZATION;

    try {
      layer._finalize();

      layer.lifecycle = LIFECYCLE.FINALIZED;
    } catch (err) {
      this._handleError('finalization', err, layer);
    }
  }

}
//# sourceMappingURL=layer-manager.js.map