import { COORDINATE_SYSTEM } from './constants';
import AttributeManager from './attribute/attribute-manager';
import UniformTransitionManager from './uniform-transition-manager';
import { diffProps, validateProps } from '../lifecycle/props';
import { count } from '../utils/count';
import log from '../utils/log';
import debug from '../debug';
import { withParameters, setParameters } from '@luma.gl/core';
import assert from '../utils/assert';
import memoize from '../utils/memoize';
import { mergeShaders } from '../utils/shader';
import { projectPosition, getWorldPosition } from '../shaderlib/project/project-functions';
import typedArrayManager from '../utils/typed-array-manager';
import Component from '../lifecycle/component';
import LayerState from './layer-state';
import { worldToPixels } from '@math.gl/web-mercator';
import { load } from '@loaders.gl/core';
const TRACE_CHANGE_FLAG = 'layer.changeFlag';
const TRACE_INITIALIZE = 'layer.initialize';
const TRACE_UPDATE = 'layer.update';
const TRACE_FINALIZE = 'layer.finalize';
const TRACE_MATCHED = 'layer.matched';
const EMPTY_ARRAY = Object.freeze([]);
const areViewportsEqual = memoize(({
  oldViewport,
  viewport
}) => {
  return oldViewport.equals(viewport);
});
let pickingColorCache = new Uint8ClampedArray(0);
const defaultProps = {
  data: {
    type: 'data',
    value: EMPTY_ARRAY,
    async: true
  },
  dataComparator: null,
  _dataDiff: {
    type: 'function',
    value: data => data && data.__diff,
    compare: false,
    optional: true
  },
  dataTransform: {
    type: 'function',
    value: null,
    compare: false,
    optional: true
  },
  onDataLoad: {
    type: 'function',
    value: null,
    compare: false,
    optional: true
  },
  fetch: {
    type: 'function',
    value: (url, {
      propName,
      layer
    }) => {
      const {
        resourceManager
      } = layer.context;
      const loadOptions = layer.getLoadOptions();
      let inResourceManager = resourceManager.contains(url);

      if (!inResourceManager && !loadOptions) {
        resourceManager.add({
          resourceId: url,
          data: url,
          persistent: false
        });
        inResourceManager = true;
      }

      if (inResourceManager) {
        return resourceManager.subscribe({
          resourceId: url,
          onChange: data => layer.internalState.reloadAsyncProp(propName, data),
          consumerId: layer.id,
          requestId: propName
        });
      }

      return load(url, loadOptions);
    },
    compare: false
  },
  updateTriggers: {},
  visible: true,
  pickable: false,
  opacity: {
    type: 'number',
    min: 0,
    max: 1,
    value: 1
  },
  onHover: {
    type: 'function',
    value: null,
    compare: false,
    optional: true
  },
  onClick: {
    type: 'function',
    value: null,
    compare: false,
    optional: true
  },
  onDragStart: {
    type: 'function',
    value: null,
    compare: false,
    optional: true
  },
  onDrag: {
    type: 'function',
    value: null,
    compare: false,
    optional: true
  },
  onDragEnd: {
    type: 'function',
    value: null,
    compare: false,
    optional: true
  },
  coordinateSystem: COORDINATE_SYSTEM.DEFAULT,
  coordinateOrigin: {
    type: 'array',
    value: [0, 0, 0],
    compare: true
  },
  modelMatrix: {
    type: 'array',
    value: null,
    compare: true,
    optional: true
  },
  wrapLongitude: false,
  positionFormat: 'XYZ',
  colorFormat: 'RGBA',
  parameters: {},
  uniforms: {},
  extensions: [],
  getPolygonOffset: {
    type: 'function',
    value: ({
      layerIndex
    }) => [0, -layerIndex * 100],
    compare: false
  },
  highlightedObjectIndex: null,
  autoHighlight: false,
  highlightColor: {
    type: 'accessor',
    value: [0, 0, 128, 128]
  }
};
export default class Layer extends Component {
  toString() {
    const className = this.constructor.layerName || this.constructor.name;
    return "".concat(className, "({id: '").concat(this.props.id, "'})");
  }

  setState(updateObject) {
    this.setChangeFlags({
      stateChanged: true
    });
    Object.assign(this.state, updateObject);
    this.setNeedsRedraw();
  }

  setNeedsRedraw(redraw = true) {
    if (this.internalState) {
      this.internalState.needsRedraw = redraw;
    }
  }

  setNeedsUpdate() {
    this.context.layerManager.setNeedsUpdate(String(this));
    this.internalState.needsUpdate = true;
  }

  getNeedsRedraw(opts = {
    clearRedrawFlags: false
  }) {
    return this._getNeedsRedraw(opts);
  }

  needsUpdate() {
    return this.internalState.needsUpdate || this.hasUniformTransition() || this.shouldUpdateState(this._getUpdateParams());
  }

  hasUniformTransition() {
    return this.internalState.uniformTransitions.active;
  }

  get isLoaded() {
    return this.internalState && !this.internalState.isAsyncPropLoading();
  }

  get wrapLongitude() {
    return this.props.wrapLongitude;
  }

  isPickable() {
    return this.props.pickable && this.props.visible;
  }

  getModels() {
    return this.state && (this.state.models || (this.state.model ? [this.state.model] : []));
  }

  getAttributeManager() {
    return this.internalState && this.internalState.attributeManager;
  }

  getCurrentLayer() {
    return this.internalState && this.internalState.layer;
  }

  getLoadOptions() {
    return this.props.loadOptions;
  }

  project(xyz) {
    const {
      viewport
    } = this.context;
    const worldPosition = getWorldPosition(xyz, {
      viewport,
      modelMatrix: this.props.modelMatrix,
      coordinateOrigin: this.props.coordinateOrigin,
      coordinateSystem: this.props.coordinateSystem
    });
    const [x, y, z] = worldToPixels(worldPosition, viewport.pixelProjectionMatrix);
    return xyz.length === 2 ? [x, y] : [x, y, z];
  }

  unproject(xy) {
    const {
      viewport
    } = this.context;
    return viewport.unproject(xy);
  }

  projectPosition(xyz) {
    return projectPosition(xyz, {
      viewport: this.context.viewport,
      modelMatrix: this.props.modelMatrix,
      coordinateOrigin: this.props.coordinateOrigin,
      coordinateSystem: this.props.coordinateSystem
    });
  }

  use64bitPositions() {
    const {
      coordinateSystem
    } = this.props;
    return coordinateSystem === COORDINATE_SYSTEM.DEFAULT || coordinateSystem === COORDINATE_SYSTEM.LNGLAT || coordinateSystem === COORDINATE_SYSTEM.CARTESIAN;
  }

  onHover(info, pickingEvent) {
    if (this.props.onHover) {
      return this.props.onHover(info, pickingEvent);
    }

    return false;
  }

  onClick(info, pickingEvent) {
    if (this.props.onClick) {
      return this.props.onClick(info, pickingEvent);
    }

    return false;
  }

  nullPickingColor() {
    return [0, 0, 0];
  }

  encodePickingColor(i, target = []) {
    target[0] = i + 1 & 255;
    target[1] = i + 1 >> 8 & 255;
    target[2] = i + 1 >> 8 >> 8 & 255;
    return target;
  }

  decodePickingColor(color) {
    assert(color instanceof Uint8Array);
    const [i1, i2, i3] = color;
    const index = i1 + i2 * 256 + i3 * 65536 - 1;
    return index;
  }

  initializeState() {
    throw new Error("Layer ".concat(this, " has not defined initializeState"));
  }

  getShaders(shaders) {
    for (const extension of this.props.extensions) {
      shaders = mergeShaders(shaders, extension.getShaders.call(this, extension));
    }

    return shaders;
  }

  shouldUpdateState({
    oldProps,
    props,
    context,
    changeFlags
  }) {
    return changeFlags.propsOrDataChanged;
  }

  updateState({
    oldProps,
    props,
    context,
    changeFlags
  }) {
    const attributeManager = this.getAttributeManager();

    if (changeFlags.dataChanged && attributeManager) {
      const {
        dataChanged
      } = changeFlags;

      if (Array.isArray(dataChanged)) {
        for (const dataRange of dataChanged) {
          attributeManager.invalidateAll(dataRange);
        }
      } else {
        attributeManager.invalidateAll();
      }
    }
  }

  finalizeState() {
    for (const model of this.getModels()) {
      model.delete();
    }

    const attributeManager = this.getAttributeManager();

    if (attributeManager) {
      attributeManager.finalize();
    }

    this.context.resourceManager.unsubscribe({
      consumerId: this.id
    });
    this.internalState.uniformTransitions.clear();
  }

  draw(opts) {
    for (const model of this.getModels()) {
      model.draw(opts);
    }
  }

  getPickingInfo({
    info,
    mode
  }) {
    const {
      index
    } = info;

    if (index >= 0) {
      if (Array.isArray(this.props.data)) {
        info.object = this.props.data[index];
      }
    }

    return info;
  }

  activateViewport(viewport) {
    const oldViewport = this.internalState.viewport;
    this.internalState.viewport = viewport;

    if (!oldViewport || !areViewportsEqual({
      oldViewport,
      viewport
    })) {
      this.setChangeFlags({
        viewportChanged: true
      });

      this._update();
    }
  }

  invalidateAttribute(name = 'all', diffReason = '') {
    const attributeManager = this.getAttributeManager();

    if (!attributeManager) {
      return;
    }

    if (name === 'all') {
      attributeManager.invalidateAll();
    } else {
      attributeManager.invalidate(name);
    }
  }

  updateAttributes(changedAttributes) {
    for (const model of this.getModels()) {
      this._setModelAttributes(model, changedAttributes);
    }
  }

  _updateAttributes(props) {
    const attributeManager = this.getAttributeManager();

    if (!attributeManager) {
      return;
    }

    const numInstances = this.getNumInstances(props);
    const startIndices = this.getStartIndices(props);
    attributeManager.update({
      data: props.data,
      numInstances,
      startIndices,
      props,
      transitions: props.transitions,
      buffers: props.data.attributes,
      context: this,
      ignoreUnknownAttributes: true
    });
    const changedAttributes = attributeManager.getChangedAttributes({
      clearChangedFlags: true
    });
    this.updateAttributes(changedAttributes);
  }

  _updateAttributeTransition() {
    const attributeManager = this.getAttributeManager();

    if (attributeManager) {
      attributeManager.updateTransition();
    }
  }

  _updateUniformTransition() {
    const {
      uniformTransitions
    } = this.internalState;

    if (uniformTransitions.active) {
      const propsInTransition = uniformTransitions.update();
      const props = Object.create(this.props);

      for (const key in propsInTransition) {
        Object.defineProperty(props, key, {
          value: propsInTransition[key]
        });
      }

      return props;
    }

    return this.props;
  }

  calculateInstancePickingColors(attribute, {
    numInstances
  }) {
    const cacheSize = pickingColorCache.length / 3;

    if (cacheSize < numInstances) {
      pickingColorCache = typedArrayManager.allocate(pickingColorCache, numInstances, {
        size: 3,
        copy: true
      });
      const newCacheSize = pickingColorCache.length / 3;
      const pickingColor = [];
      assert(newCacheSize < 16777215, 'index out of picking color range');

      for (let i = cacheSize; i < newCacheSize; i++) {
        this.encodePickingColor(i, pickingColor);
        pickingColorCache[i * 3 + 0] = pickingColor[0];
        pickingColorCache[i * 3 + 1] = pickingColor[1];
        pickingColorCache[i * 3 + 2] = pickingColor[2];
      }
    }

    attribute.value = pickingColorCache.subarray(0, numInstances * 3);
  }

  _setModelAttributes(model, changedAttributes) {
    const attributeManager = this.getAttributeManager();
    const excludeAttributes = model.userData.excludeAttributes || {};
    const shaderAttributes = attributeManager.getShaderAttributes(changedAttributes, excludeAttributes);
    model.setAttributes(shaderAttributes);
  }

  clearPickingColor(color) {
    const {
      pickingColors,
      instancePickingColors
    } = this.getAttributeManager().attributes;
    const colors = pickingColors || instancePickingColors;
    const i = this.decodePickingColor(color);
    const start = colors.getVertexOffset(i);
    const end = colors.getVertexOffset(i + 1);
    colors.buffer.subData({
      data: new Uint8Array(end - start),
      offset: start
    });
  }

  restorePickingColors() {
    const {
      pickingColors,
      instancePickingColors
    } = this.getAttributeManager().attributes;
    const colors = pickingColors || instancePickingColors;
    colors.updateSubBuffer({
      startOffset: 0
    });
  }

  getNumInstances(props) {
    props = props || this.props;

    if (props.numInstances !== undefined) {
      return props.numInstances;
    }

    if (this.state && this.state.numInstances !== undefined) {
      return this.state.numInstances;
    }

    return count(props.data);
  }

  getStartIndices(props) {
    props = props || this.props;

    if (props.startIndices !== undefined) {
      return props.startIndices;
    }

    if (this.state && this.state.startIndices) {
      return this.state.startIndices;
    }

    return null;
  }

  _initialize() {
    debug(TRACE_INITIALIZE, this);

    this._initState();

    this.initializeState(this.context);

    for (const extension of this.props.extensions) {
      extension.initializeState.call(this, this.context, extension);
    }

    this.setChangeFlags({
      dataChanged: true,
      propsChanged: true,
      viewportChanged: true,
      extensionsChanged: true
    });

    this._updateState();
  }

  _update() {
    const stateNeedsUpdate = this.needsUpdate();
    debug(TRACE_UPDATE, this, stateNeedsUpdate);

    if (stateNeedsUpdate) {
      this._updateState();
    }
  }

  _updateState() {
    const currentProps = this.props;
    const currentViewport = this.context.viewport;

    const propsInTransition = this._updateUniformTransition();

    this.internalState.propsInTransition = propsInTransition;
    this.context.viewport = this.internalState.viewport || currentViewport;
    this.props = propsInTransition;

    const updateParams = this._getUpdateParams();

    if (this.context.gl) {
      this.updateState(updateParams);
    } else {
      try {
        this.updateState(updateParams);
      } catch (error) {}
    }

    for (const extension of this.props.extensions) {
      extension.updateState.call(this, updateParams, extension);
    }

    this._updateModules(updateParams);

    if (this.isComposite) {
      this._renderLayers(updateParams);
    } else {
      this.setNeedsRedraw();

      this._updateAttributes(this.props);

      if (this.state.model) {
        this.state.model.setInstanceCount(this.getNumInstances());
      }
    }

    this.context.viewport = currentViewport;
    this.props = currentProps;
    this.clearChangeFlags();
    this.internalState.needsUpdate = false;
    this.internalState.resetOldProps();
  }

  _finalize() {
    debug(TRACE_FINALIZE, this);
    assert(this.internalState && this.state);
    this.finalizeState(this.context);

    for (const extension of this.props.extensions) {
      extension.finalizeState.call(this, extension);
    }
  }

  drawLayer({
    moduleParameters = null,
    uniforms = {},
    parameters = {}
  }) {
    this._updateAttributeTransition();

    const currentProps = this.props;
    this.props = this.internalState.propsInTransition || currentProps;
    const {
      opacity
    } = this.props;
    uniforms.opacity = Math.pow(opacity, 1 / 2.2);

    if (moduleParameters) {
      this.setModuleParameters(moduleParameters);
    }

    const {
      getPolygonOffset
    } = this.props;
    const offsets = getPolygonOffset && getPolygonOffset(uniforms) || [0, 0];
    setParameters(this.context.gl, {
      polygonOffset: offsets
    });
    withParameters(this.context.gl, parameters, () => {
      const opts = {
        moduleParameters,
        uniforms,
        parameters,
        context: this.context
      };

      for (const extension of this.props.extensions) {
        extension.draw.call(this, opts, extension);
      }

      this.draw(opts);
    });
    this.props = currentProps;
  }

  getChangeFlags() {
    return this.internalState.changeFlags;
  }

  setChangeFlags(flags) {
    const {
      changeFlags
    } = this.internalState;

    for (const key in changeFlags) {
      if (flags[key] && !changeFlags[key]) {
        changeFlags[key] = flags[key];
        debug(TRACE_CHANGE_FLAG, this, key, flags);
      }
    }

    const propsOrDataChanged = changeFlags.dataChanged || changeFlags.updateTriggersChanged || changeFlags.propsChanged || changeFlags.extensionsChanged;
    changeFlags.propsOrDataChanged = propsOrDataChanged;
    changeFlags.somethingChanged = propsOrDataChanged || flags.viewportChanged || flags.stateChanged;
  }

  clearChangeFlags() {
    this.internalState.changeFlags = {
      dataChanged: false,
      propsChanged: false,
      updateTriggersChanged: false,
      viewportChanged: false,
      stateChanged: false,
      extensionsChanged: false,
      propsOrDataChanged: false,
      somethingChanged: false
    };
  }

  diffProps(newProps, oldProps) {
    const changeFlags = diffProps(newProps, oldProps);

    if (changeFlags.updateTriggersChanged) {
      for (const key in changeFlags.updateTriggersChanged) {
        if (changeFlags.updateTriggersChanged[key]) {
          this.invalidateAttribute(key);
        }
      }
    }

    if (changeFlags.transitionsChanged) {
      for (const key in changeFlags.transitionsChanged) {
        this.internalState.uniformTransitions.add(key, oldProps[key], newProps[key], newProps.transitions[key]);
      }
    }

    return this.setChangeFlags(changeFlags);
  }

  validateProps() {
    validateProps(this.props);
  }

  setModuleParameters(moduleParameters) {
    for (const model of this.getModels()) {
      model.updateModuleSettings(moduleParameters);
    }
  }

  _updateModules({
    props,
    oldProps
  }) {
    const {
      autoHighlight,
      highlightedObjectIndex,
      highlightColor
    } = props;

    if (oldProps.autoHighlight !== autoHighlight || oldProps.highlightedObjectIndex !== highlightedObjectIndex || oldProps.highlightColor !== highlightColor) {
      const parameters = {};

      if (!autoHighlight) {
        parameters.pickingSelectedColor = null;
      }

      if (Array.isArray(highlightColor)) {
        parameters.pickingHighlightColor = highlightColor;
      }

      if (Number.isInteger(highlightedObjectIndex)) {
        parameters.pickingSelectedColor = highlightedObjectIndex >= 0 ? this.encodePickingColor(highlightedObjectIndex) : null;
      }

      this.setModuleParameters(parameters);
    }
  }

  _getUpdateParams() {
    return {
      props: this.props,
      oldProps: this.internalState.getOldProps(),
      context: this.context,
      changeFlags: this.internalState.changeFlags
    };
  }

  _getNeedsRedraw(opts) {
    if (!this.internalState) {
      return false;
    }

    let redraw = false;
    redraw = redraw || this.internalState.needsRedraw && this.id;
    this.internalState.needsRedraw = this.internalState.needsRedraw && !opts.clearRedrawFlags;
    const attributeManager = this.getAttributeManager();
    const attributeManagerNeedsRedraw = attributeManager && attributeManager.getNeedsRedraw(opts);
    redraw = redraw || attributeManagerNeedsRedraw;
    return redraw;
  }

  _getAttributeManager() {
    return new AttributeManager(this.context.gl, {
      id: this.props.id,
      stats: this.context.stats,
      timeline: this.context.timeline
    });
  }

  _initState() {
    assert(!this.internalState && !this.state);
    assert(isFinite(this.props.coordinateSystem), "".concat(this.id, ": invalid coordinateSystem"));

    const attributeManager = this._getAttributeManager();

    if (attributeManager) {
      attributeManager.addInstanced({
        instancePickingColors: {
          type: 5121,
          size: 3,
          noAlloc: true,
          update: this.calculateInstancePickingColors
        }
      });
    }

    this.internalState = new LayerState({
      attributeManager,
      layer: this
    });
    this.clearChangeFlags();
    this.state = {};
    Object.defineProperty(this.state, 'attributeManager', {
      get: () => {
        log.deprecated('layer.state.attributeManager', 'layer.getAttributeManager()');
        return attributeManager;
      }
    });
    this.internalState.layer = this;
    this.internalState.uniformTransitions = new UniformTransitionManager(this.context.timeline);
    this.internalState.onAsyncPropUpdated = this._onAsyncPropUpdated.bind(this);
    this.internalState.setAsyncProps(this.props);
  }

  _transferState(oldLayer) {
    debug(TRACE_MATCHED, this, this === oldLayer);
    const {
      state,
      internalState
    } = oldLayer;
    assert(state && internalState);

    if (this === oldLayer) {
      return;
    }

    this.internalState = internalState;
    this.internalState.layer = this;
    this.state = state;
    this.internalState.setAsyncProps(this.props);
    this.diffProps(this.props, this.internalState.getOldProps());
  }

  _onAsyncPropUpdated() {
    this.diffProps(this.props, this.internalState.getOldProps());
    this.setNeedsUpdate();
  }

}
Layer.layerName = 'Layer';
Layer.defaultProps = defaultProps;
//# sourceMappingURL=layer.js.map