import log from '../utils/log';
import { isAsyncIterable } from '../utils/iterable-utils';
import { parsePropTypes } from './prop-types';
import { PROP_SYMBOLS } from './constants';
const {
  COMPONENT,
  ASYNC_ORIGINAL,
  ASYNC_RESOLVED,
  ASYNC_DEFAULTS
} = PROP_SYMBOLS;
export function createProps() {
  const component = this;
  const propsPrototype = getPropsPrototype(component.constructor);
  const propsInstance = Object.create(propsPrototype);
  propsInstance[COMPONENT] = component;
  propsInstance[ASYNC_ORIGINAL] = {};
  propsInstance[ASYNC_RESOLVED] = {};

  for (let i = 0; i < arguments.length; ++i) {
    const props = arguments[i];

    for (const key in props) {
      propsInstance[key] = props[key];
    }
  }

  Object.freeze(propsInstance);
  return propsInstance;
}

function getPropsPrototype(componentClass) {
  const defaultProps = getOwnProperty(componentClass, '_mergedDefaultProps');

  if (!defaultProps) {
    createPropsPrototypeAndTypes(componentClass);
    return componentClass._mergedDefaultProps;
  }

  return defaultProps;
}

function createPropsPrototypeAndTypes(componentClass) {
  const parent = componentClass.prototype;

  if (!parent) {
    return;
  }

  const parentClass = Object.getPrototypeOf(componentClass);
  const parentDefaultProps = getPropsPrototype(parentClass);
  const componentDefaultProps = getOwnProperty(componentClass, 'defaultProps') || {};
  const componentPropDefs = parsePropTypes(componentDefaultProps);
  const defaultProps = createPropsPrototype(componentPropDefs.defaultProps, parentDefaultProps, componentClass);
  const propTypes = Object.assign({}, parentClass._propTypes, componentPropDefs.propTypes);
  addAsyncPropsToPropPrototype(defaultProps, propTypes);
  const deprecatedProps = Object.assign({}, parentClass._deprecatedProps, componentPropDefs.deprecatedProps);
  addDeprecatedPropsToPropPrototype(defaultProps, deprecatedProps);
  componentClass._mergedDefaultProps = defaultProps;
  componentClass._propTypes = propTypes;
  componentClass._deprecatedProps = deprecatedProps;
}

function createPropsPrototype(props, parentProps, componentClass) {
  const defaultProps = Object.create(null);
  Object.assign(defaultProps, parentProps, props);
  const id = getComponentName(componentClass);
  delete props.id;
  Object.defineProperties(defaultProps, {
    id: {
      writable: true,
      value: id
    }
  });
  return defaultProps;
}

function addDeprecatedPropsToPropPrototype(defaultProps, deprecatedProps) {
  for (const propName in deprecatedProps) {
    Object.defineProperty(defaultProps, propName, {
      enumerable: false,

      set(newValue) {
        const nameStr = "".concat(this.id, ": ").concat(propName);

        for (const newPropName of deprecatedProps[propName]) {
          if (!hasOwnProperty(this, newPropName)) {
            this[newPropName] = newValue;
          }
        }

        log.deprecated(nameStr, deprecatedProps[propName].join('/'))();
      }

    });
  }
}

function addAsyncPropsToPropPrototype(defaultProps, propTypes) {
  const defaultValues = {};
  const descriptors = {};

  for (const propName in propTypes) {
    const propType = propTypes[propName];
    const {
      name,
      value
    } = propType;

    if (propType.async) {
      defaultValues[name] = value;
      descriptors[name] = getDescriptorForAsyncProp(name, value);
    }
  }

  defaultProps[ASYNC_DEFAULTS] = defaultValues;
  defaultProps[ASYNC_ORIGINAL] = {};
  Object.defineProperties(defaultProps, descriptors);
}

function getDescriptorForAsyncProp(name) {
  return {
    enumerable: true,

    set(newValue) {
      if (typeof newValue === 'string' || newValue instanceof Promise || isAsyncIterable(newValue)) {
        this[ASYNC_ORIGINAL][name] = newValue;
      } else {
        this[ASYNC_RESOLVED][name] = newValue;
      }
    },

    get() {
      if (this[ASYNC_RESOLVED]) {
        if (name in this[ASYNC_RESOLVED]) {
          const value = this[ASYNC_RESOLVED][name];
          return value || this[ASYNC_DEFAULTS][name];
        }

        if (name in this[ASYNC_ORIGINAL]) {
          const state = this[COMPONENT] && this[COMPONENT].internalState;

          if (state && state.hasAsyncProp(name)) {
            return state.getAsyncProp(name) || this[ASYNC_DEFAULTS][name];
          }
        }
      }

      return this[ASYNC_DEFAULTS][name];
    }

  };
}

function hasOwnProperty(object, prop) {
  return Object.prototype.hasOwnProperty.call(object, prop);
}

function getOwnProperty(object, prop) {
  return hasOwnProperty(object, prop) && object[prop];
}

function getComponentName(componentClass) {
  const componentName = getOwnProperty(componentClass, 'layerName') || getOwnProperty(componentClass, 'componentName');

  if (!componentName) {
    log.once(0, "".concat(componentClass.name, ".componentName not specified"))();
  }

  return componentName || componentClass.name;
}
//# sourceMappingURL=create-props.js.map