import Pass from './pass';
import { clear, setParameters, withParameters, cssToDeviceRatio } from '@luma.gl/core';
import log from '../utils/log';
export default class LayersPass extends Pass {
  render(props) {
    const gl = this.gl;
    setParameters(gl, {
      framebuffer: props.target
    });
    return this._drawLayers(props);
  }

  _drawLayers(props) {
    const {
      viewports,
      views,
      onViewportActive,
      clearCanvas = true
    } = props;
    const gl = this.gl;

    if (clearCanvas) {
      clearGLCanvas(gl);
    }

    const renderStats = [];

    for (const viewportOrDescriptor of viewports) {
      const viewport = viewportOrDescriptor.viewport || viewportOrDescriptor;
      const view = views && views[viewport.id];
      onViewportActive(viewport);

      const drawLayerParams = this._getDrawLayerParams(viewport, props);

      props.view = view;
      const subViewports = viewport.subViewports || [viewport];

      for (const subViewport of subViewports) {
        props.viewport = subViewport;

        const stats = this._drawLayersInViewport(gl, props, drawLayerParams);

        renderStats.push(stats);
      }
    }

    return renderStats;
  }

  _getDrawLayerParams(viewport, {
    layers,
    pass = 'unknown',
    layerFilter,
    effects,
    moduleParameters
  }) {
    const drawLayerParams = [];
    const indexResolver = layerIndexResolver();

    for (let layerIndex = 0; layerIndex < layers.length; layerIndex++) {
      const layer = layers[layerIndex];

      const shouldDrawLayer = this._shouldDrawLayer(layer, viewport, pass, layerFilter);

      const layerRenderIndex = indexResolver(layer, shouldDrawLayer);
      const layerParam = {
        shouldDrawLayer,
        layerRenderIndex
      };

      if (shouldDrawLayer) {
        layerParam.moduleParameters = this._getModuleParameters(layer, effects, pass, moduleParameters);
        layerParam.layerParameters = this.getLayerParameters(layer, layerIndex);
      }

      drawLayerParams[layerIndex] = layerParam;
    }

    return drawLayerParams;
  }

  _drawLayersInViewport(gl, {
    layers,
    onError,
    viewport,
    view
  }, drawLayerParams) {
    const glViewport = getGLViewport(gl, {
      viewport
    });

    if (view && view.props.clear) {
      const clearOpts = view.props.clear === true ? {
        color: true,
        depth: true
      } : view.props.clear;
      withParameters(gl, {
        scissorTest: true,
        scissor: glViewport
      }, () => clear(gl, clearOpts));
    }

    const renderStatus = {
      totalCount: layers.length,
      visibleCount: 0,
      compositeCount: 0,
      pickableCount: 0
    };
    setParameters(gl, {
      viewport: glViewport
    });

    for (let layerIndex = 0; layerIndex < layers.length; layerIndex++) {
      const layer = layers[layerIndex];
      const {
        shouldDrawLayer,
        layerRenderIndex,
        moduleParameters,
        layerParameters
      } = drawLayerParams[layerIndex];

      if (shouldDrawLayer && layer.props.pickable) {
        renderStatus.pickableCount++;
      }

      if (layer.isComposite) {
        renderStatus.compositeCount++;
      } else if (shouldDrawLayer) {
        renderStatus.visibleCount++;
        moduleParameters.viewport = viewport;

        try {
          layer.drawLayer({
            moduleParameters,
            uniforms: {
              layerIndex: layerRenderIndex
            },
            parameters: layerParameters
          });
        } catch (err) {
          if (onError) {
            onError(err, layer);
          } else {
            log.error("error during drawing of ".concat(layer), err)();
          }
        }
      }
    }

    return renderStatus;
  }

  shouldDrawLayer(layer) {
    return true;
  }

  getModuleParameters(layer, effects) {
    return null;
  }

  getLayerParameters(layer, layerIndex) {
    return layer.props.parameters;
  }

  _shouldDrawLayer(layer, viewport, pass, layerFilter) {
    let shouldDrawLayer = this.shouldDrawLayer(layer) && layer.props.visible;

    if (shouldDrawLayer && layerFilter) {
      shouldDrawLayer = layerFilter({
        layer,
        viewport,
        isPicking: pass.startsWith('picking'),
        renderPass: pass
      });
    }

    if (shouldDrawLayer) {
      layer.activateViewport(viewport);
    }

    return shouldDrawLayer;
  }

  _getModuleParameters(layer, effects, pass, overrides) {
    const moduleParameters = Object.assign(Object.create(layer.props), {
      autoWrapLongitude: layer.wrapLongitude,
      viewport: layer.context.viewport,
      mousePosition: layer.context.mousePosition,
      pickingActive: 0,
      devicePixelRatio: cssToDeviceRatio(this.gl)
    });

    if (effects) {
      for (const effect of effects) {
        Object.assign(moduleParameters, effect.getModuleParameters(layer));
      }
    }

    return Object.assign(moduleParameters, this.getModuleParameters(layer, effects), overrides);
  }

}
export function layerIndexResolver(startIndex = 0, layerIndices = {}) {
  const resolvers = {};
  return (layer, isDrawn) => {
    const indexOverride = layer.props._offset;
    const layerId = layer.id;
    const parentId = layer.parent && layer.parent.id;
    let index;

    if (parentId in resolvers) {
      const resolver = resolvers[parentId] = resolvers[parentId] || layerIndexResolver(layerIndices[parentId], layerIndices);
      index = resolver(layer, isDrawn);
      resolvers[layerId] = resolver;
    } else if (Number.isFinite(indexOverride)) {
      index = indexOverride + (layerIndices[parentId] || 0);
      resolvers[layerId] = null;
    } else {
      index = startIndex;
    }

    if (isDrawn && index >= startIndex) {
      startIndex = index + 1;
    }

    layerIndices[layerId] = index;
    return index;
  };
}

function getGLViewport(gl, {
  viewport
}) {
  const height = gl.canvas ? gl.canvas.clientHeight || gl.canvas.height : 100;
  const dimensions = viewport;
  const pixelRatio = cssToDeviceRatio(gl);
  return [dimensions.x * pixelRatio, (height - dimensions.y - dimensions.height) * pixelRatio, dimensions.width * pixelRatio, dimensions.height * pixelRatio];
}

function clearGLCanvas(gl) {
  const width = gl.drawingBufferWidth;
  const height = gl.drawingBufferHeight;
  setParameters(gl, {
    viewport: [0, 0, width, height]
  });
  gl.clear(16384 | 256);
}
//# sourceMappingURL=layers-pass.js.map