import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import _classCallCheck from "@babel/runtime/helpers/esm/classCallCheck";
import _createClass from "@babel/runtime/helpers/esm/createClass";
import { Framebuffer, Texture2D, isWebGL2, readPixelsToArray, cssToDeviceRatio, cssToDevicePixels } from '@luma.gl/core';
import assert from '../utils/assert';
import log from '../utils/log';
import PickLayersPass from '../passes/pick-layers-pass';
import { getClosestObject, getUniqueObjects } from './picking/query-object';
import { processPickInfo, getLayerPickingInfo } from './picking/pick-info';

var DeckPicker = function () {
  function DeckPicker(gl) {
    _classCallCheck(this, DeckPicker);

    this.gl = gl;
    this.pickingFBO = null;
    this.pickLayersPass = new PickLayersPass(gl);
    this.layerFilter = null;
    this.lastPickedInfo = {
      index: -1,
      layerId: null,
      info: null
    };
    this._onError = null;
  }

  _createClass(DeckPicker, [{
    key: "setProps",
    value: function setProps(props) {
      if ('layerFilter' in props) {
        this.layerFilter = props.layerFilter;
      }

      if ('onError' in props) {
        this._onError = props.onError;
      }
    }
  }, {
    key: "finalize",
    value: function finalize() {
      if (this.pickingFBO) {
        this.pickingFBO["delete"]();
      }

      if (this.depthFBO) {
        this.depthFBO.color["delete"]();
        this.depthFBO["delete"]();
      }
    }
  }, {
    key: "pickObject",
    value: function pickObject(opts) {
      return this._pickClosestObject(opts);
    }
  }, {
    key: "pickObjects",
    value: function pickObjects(opts) {
      return this._pickVisibleObjects(opts);
    }
  }, {
    key: "getLastPickedObject",
    value: function getLastPickedObject(_ref) {
      var x = _ref.x,
          y = _ref.y,
          layers = _ref.layers,
          viewports = _ref.viewports;
      var lastPickedInfo = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : this.lastPickedInfo.info;
      var lastPickedLayerId = lastPickedInfo && lastPickedInfo.layer && lastPickedInfo.layer.id;
      var layer = lastPickedLayerId ? layers.find(function (l) {
        return l.id === lastPickedLayerId;
      }) : null;
      var coordinate = viewports[0] && viewports[0].unproject([x, y]);
      var info = {
        x: x,
        y: y,
        coordinate: coordinate,
        lngLat: coordinate,
        layer: layer
      };

      if (layer) {
        return Object.assign({}, lastPickedInfo, info);
      }

      return Object.assign(info, {
        color: null,
        object: null,
        index: -1
      });
    }
  }, {
    key: "_resizeBuffer",
    value: function _resizeBuffer() {
      var gl = this.gl;

      if (!this.pickingFBO) {
        this.pickingFBO = new Framebuffer(gl);

        if (Framebuffer.isSupported(gl, {
          colorBufferFloat: true
        })) {
          this.depthFBO = new Framebuffer(gl);
          this.depthFBO.attach(_defineProperty({}, 36064, new Texture2D(gl, {
            format: isWebGL2(gl) ? 34836 : 6408,
            type: 5126
          })));
        }
      }

      this.pickingFBO.resize({
        width: gl.canvas.width,
        height: gl.canvas.height
      });

      if (this.depthFBO) {
        this.depthFBO.resize({
          width: gl.canvas.width,
          height: gl.canvas.height
        });
      }

      return this.pickingFBO;
    }
  }, {
    key: "_getPickable",
    value: function _getPickable(layers) {
      var pickableLayers = layers.filter(function (layer) {
        return layer.isPickable() && !layer.isComposite;
      });

      if (pickableLayers.length > 255) {
        log.warn('Too many pickable layers, only picking the first 255')();
        return pickableLayers.slice(0, 255);
      }

      return pickableLayers;
    }
  }, {
    key: "_pickClosestObject",
    value: function _pickClosestObject(_ref2) {
      var layers = _ref2.layers,
          viewports = _ref2.viewports,
          x = _ref2.x,
          y = _ref2.y,
          _ref2$radius = _ref2.radius,
          radius = _ref2$radius === void 0 ? 0 : _ref2$radius,
          _ref2$depth = _ref2.depth,
          depth = _ref2$depth === void 0 ? 1 : _ref2$depth,
          _ref2$mode = _ref2.mode,
          mode = _ref2$mode === void 0 ? 'query' : _ref2$mode,
          unproject3D = _ref2.unproject3D,
          onViewportActive = _ref2.onViewportActive;
      layers = this._getPickable(layers);

      this._resizeBuffer();

      var pixelRatio = cssToDeviceRatio(this.gl);
      var devicePixelRange = cssToDevicePixels(this.gl, [x, y], true);
      var devicePixel = [devicePixelRange.x + Math.floor(devicePixelRange.width / 2), devicePixelRange.y + Math.floor(devicePixelRange.height / 2)];
      var deviceRadius = Math.round(radius * pixelRatio);
      var _this$pickingFBO = this.pickingFBO,
          width = _this$pickingFBO.width,
          height = _this$pickingFBO.height;

      var deviceRect = this._getPickingRect({
        deviceX: devicePixel[0],
        deviceY: devicePixel[1],
        deviceRadius: deviceRadius,
        deviceWidth: width,
        deviceHeight: height
      });

      var infos;
      var result = [];
      var affectedLayers = {};

      for (var i = 0; i < depth; i++) {
        var pickedColors = deviceRect && this._drawAndSample({
          layers: layers,
          viewports: viewports,
          onViewportActive: onViewportActive,
          deviceRect: deviceRect,
          pass: "picking:".concat(mode),
          redrawReason: mode
        });

        var pickInfo = getClosestObject({
          pickedColors: pickedColors,
          layers: layers,
          deviceX: devicePixel[0],
          deviceY: devicePixel[1],
          deviceRadius: deviceRadius,
          deviceRect: deviceRect
        });
        var z = void 0;

        if (pickInfo.pickedLayer && unproject3D && this.depthFBO) {
          var zValues = this._drawAndSample({
            layers: [pickInfo.pickedLayer],
            viewports: viewports,
            onViewportActive: onViewportActive,
            deviceRect: {
              x: pickInfo.pickedX,
              y: pickInfo.pickedY,
              width: 1,
              height: 1
            },
            pass: "picking:".concat(mode),
            redrawReason: 'pick-z',
            pickZ: true
          });

          z = zValues[0] * viewports[0].distanceScales.metersPerUnit[2] + viewports[0].position[2];
        }

        if (pickInfo.pickedColor && i + 1 < depth) {
          var layerId = pickInfo.pickedColor[3] - 1;
          affectedLayers[layerId] = true;
          layers[layerId].clearPickingColor(pickInfo.pickedColor);
        }

        infos = processPickInfo({
          pickInfo: pickInfo,
          lastPickedInfo: this.lastPickedInfo,
          mode: mode,
          layers: layers,
          viewports: viewports,
          x: x,
          y: y,
          z: z,
          pixelRatio: pixelRatio
        });
        var _iteratorNormalCompletion = true;
        var _didIteratorError = false;
        var _iteratorError = undefined;

        try {
          for (var _iterator = infos.values()[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
            var info = _step.value;

            if (info.layer) {
              result.push(info);
            }
          }
        } catch (err) {
          _didIteratorError = true;
          _iteratorError = err;
        } finally {
          try {
            if (!_iteratorNormalCompletion && _iterator["return"] != null) {
              _iterator["return"]();
            }
          } finally {
            if (_didIteratorError) {
              throw _iteratorError;
            }
          }
        }

        if (!pickInfo.pickedColor) {
          break;
        }
      }

      for (var _layerId in affectedLayers) {
        layers[_layerId].restorePickingColors();
      }

      return {
        result: result,
        emptyInfo: infos && infos.get(null)
      };
    }
  }, {
    key: "_pickVisibleObjects",
    value: function _pickVisibleObjects(_ref3) {
      var layers = _ref3.layers,
          viewports = _ref3.viewports,
          x = _ref3.x,
          y = _ref3.y,
          _ref3$width = _ref3.width,
          width = _ref3$width === void 0 ? 1 : _ref3$width,
          _ref3$height = _ref3.height,
          height = _ref3$height === void 0 ? 1 : _ref3$height,
          _ref3$mode = _ref3.mode,
          mode = _ref3$mode === void 0 ? 'query' : _ref3$mode,
          onViewportActive = _ref3.onViewportActive;
      layers = this._getPickable(layers);

      this._resizeBuffer();

      var pixelRatio = cssToDeviceRatio(this.gl);
      var leftTop = cssToDevicePixels(this.gl, [x, y], true);
      var deviceLeft = leftTop.x;
      var deviceTop = leftTop.y + leftTop.height;
      var rightBottom = cssToDevicePixels(this.gl, [x + width, y + height], true);
      var deviceRight = rightBottom.x + rightBottom.width;
      var deviceBottom = rightBottom.y;
      var deviceRect = {
        x: deviceLeft,
        y: deviceBottom,
        width: deviceRight - deviceLeft,
        height: deviceTop - deviceBottom
      };

      var pickedColors = this._drawAndSample({
        layers: layers,
        viewports: viewports,
        onViewportActive: onViewportActive,
        deviceRect: deviceRect,
        pass: "picking:".concat(mode),
        redrawReason: mode
      });

      var pickInfos = getUniqueObjects({
        pickedColors: pickedColors,
        layers: layers
      });
      var uniqueInfos = new Map();
      pickInfos.forEach(function (pickInfo) {
        var info = {
          color: pickInfo.pickedColor,
          layer: null,
          index: pickInfo.pickedObjectIndex,
          picked: true,
          x: x,
          y: y,
          width: width,
          height: height,
          pixelRatio: pixelRatio
        };
        info = getLayerPickingInfo({
          layer: pickInfo.pickedLayer,
          info: info,
          mode: mode
        });

        if (!uniqueInfos.has(info.object)) {
          uniqueInfos.set(info.object, info);
        }
      });
      return Array.from(uniqueInfos.values());
    }
  }, {
    key: "_drawAndSample",
    value: function _drawAndSample(_ref4) {
      var layers = _ref4.layers,
          viewports = _ref4.viewports,
          onViewportActive = _ref4.onViewportActive,
          deviceRect = _ref4.deviceRect,
          pass = _ref4.pass,
          redrawReason = _ref4.redrawReason,
          pickZ = _ref4.pickZ;
      assert(deviceRect.width > 0 && deviceRect.height > 0);

      if (layers.length < 1) {
        return null;
      }

      var pickingFBO = pickZ ? this.depthFBO : this.pickingFBO;
      this.pickLayersPass.render({
        layers: layers,
        layerFilter: this.layerFilter,
        onError: this._onError,
        viewports: viewports,
        onViewportActive: onViewportActive,
        pickingFBO: pickingFBO,
        deviceRect: deviceRect,
        pass: pass,
        redrawReason: redrawReason,
        pickZ: pickZ
      });
      var x = deviceRect.x,
          y = deviceRect.y,
          width = deviceRect.width,
          height = deviceRect.height;
      var pickedColors = new (pickZ ? Float32Array : Uint8Array)(width * height * 4);
      readPixelsToArray(pickingFBO, {
        sourceX: x,
        sourceY: y,
        sourceWidth: width,
        sourceHeight: height,
        target: pickedColors
      });
      return pickedColors;
    }
  }, {
    key: "_getPickingRect",
    value: function _getPickingRect(_ref5) {
      var deviceX = _ref5.deviceX,
          deviceY = _ref5.deviceY,
          deviceRadius = _ref5.deviceRadius,
          deviceWidth = _ref5.deviceWidth,
          deviceHeight = _ref5.deviceHeight;
      var x = Math.max(0, deviceX - deviceRadius);
      var y = Math.max(0, deviceY - deviceRadius);
      var width = Math.min(deviceWidth, deviceX + deviceRadius + 1) - x;
      var height = Math.min(deviceHeight, deviceY + deviceRadius + 1) - y;

      if (width <= 0 || height <= 0) {
        return null;
      }

      return {
        x: x,
        y: y,
        width: width,
        height: height
      };
    }
  }]);

  return DeckPicker;
}();

export { DeckPicker as default };
//# sourceMappingURL=deck-picker.js.map