import log from '../utils/log';
import { isAsyncIterable } from '../utils/iterable-utils';
import { parsePropTypes } from './prop-types';
import { PROP_SYMBOLS } from './constants';
var COMPONENT = PROP_SYMBOLS.COMPONENT,
    ASYNC_ORIGINAL = PROP_SYMBOLS.ASYNC_ORIGINAL,
    ASYNC_RESOLVED = PROP_SYMBOLS.ASYNC_RESOLVED,
    ASYNC_DEFAULTS = PROP_SYMBOLS.ASYNC_DEFAULTS;
export function createProps() {
  var component = this;
  var propsPrototype = getPropsPrototype(component.constructor);
  var propsInstance = Object.create(propsPrototype);
  propsInstance[COMPONENT] = component;
  propsInstance[ASYNC_ORIGINAL] = {};
  propsInstance[ASYNC_RESOLVED] = {};

  for (var i = 0; i < arguments.length; ++i) {
    var props = arguments[i];

    for (var key in props) {
      propsInstance[key] = props[key];
    }
  }

  Object.freeze(propsInstance);
  return propsInstance;
}

function getPropsPrototype(componentClass) {
  var defaultProps = getOwnProperty(componentClass, '_mergedDefaultProps');

  if (!defaultProps) {
    createPropsPrototypeAndTypes(componentClass);
    return componentClass._mergedDefaultProps;
  }

  return defaultProps;
}

function createPropsPrototypeAndTypes(componentClass) {
  var parent = componentClass.prototype;

  if (!parent) {
    return;
  }

  var parentClass = Object.getPrototypeOf(componentClass);
  var parentDefaultProps = getPropsPrototype(parentClass);
  var componentDefaultProps = getOwnProperty(componentClass, 'defaultProps') || {};
  var componentPropDefs = parsePropTypes(componentDefaultProps);
  var defaultProps = createPropsPrototype(componentPropDefs.defaultProps, parentDefaultProps, componentClass);
  var propTypes = Object.assign({}, parentClass._propTypes, componentPropDefs.propTypes);
  addAsyncPropsToPropPrototype(defaultProps, propTypes);
  var deprecatedProps = Object.assign({}, parentClass._deprecatedProps, componentPropDefs.deprecatedProps);
  addDeprecatedPropsToPropPrototype(defaultProps, deprecatedProps);
  componentClass._mergedDefaultProps = defaultProps;
  componentClass._propTypes = propTypes;
  componentClass._deprecatedProps = deprecatedProps;
}

function createPropsPrototype(props, parentProps, componentClass) {
  var defaultProps = Object.create(null);
  Object.assign(defaultProps, parentProps, props);
  var id = getComponentName(componentClass);
  delete props.id;
  Object.defineProperties(defaultProps, {
    id: {
      writable: true,
      value: id
    }
  });
  return defaultProps;
}

function addDeprecatedPropsToPropPrototype(defaultProps, deprecatedProps) {
  var _loop = function _loop(propName) {
    Object.defineProperty(defaultProps, propName, {
      enumerable: false,
      set: function set(newValue) {
        var nameStr = "".concat(this.id, ": ").concat(propName);
        var _iteratorNormalCompletion = true;
        var _didIteratorError = false;
        var _iteratorError = undefined;

        try {
          for (var _iterator = deprecatedProps[propName][Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
            var newPropName = _step.value;

            if (!hasOwnProperty(this, newPropName)) {
              this[newPropName] = newValue;
            }
          }
        } catch (err) {
          _didIteratorError = true;
          _iteratorError = err;
        } finally {
          try {
            if (!_iteratorNormalCompletion && _iterator["return"] != null) {
              _iterator["return"]();
            }
          } finally {
            if (_didIteratorError) {
              throw _iteratorError;
            }
          }
        }

        log.deprecated(nameStr, deprecatedProps[propName].join('/'))();
      }
    });
  };

  for (var propName in deprecatedProps) {
    _loop(propName);
  }
}

function addAsyncPropsToPropPrototype(defaultProps, propTypes) {
  var defaultValues = {};
  var descriptors = {};

  for (var propName in propTypes) {
    var propType = propTypes[propName];
    var name = propType.name,
        value = propType.value;

    if (propType.async) {
      defaultValues[name] = value;
      descriptors[name] = getDescriptorForAsyncProp(name, value);
    }
  }

  defaultProps[ASYNC_DEFAULTS] = defaultValues;
  defaultProps[ASYNC_ORIGINAL] = {};
  Object.defineProperties(defaultProps, descriptors);
}

function getDescriptorForAsyncProp(name) {
  return {
    enumerable: true,
    set: function set(newValue) {
      if (typeof newValue === 'string' || newValue instanceof Promise || isAsyncIterable(newValue)) {
        this[ASYNC_ORIGINAL][name] = newValue;
      } else {
        this[ASYNC_RESOLVED][name] = newValue;
      }
    },
    get: function get() {
      if (this[ASYNC_RESOLVED]) {
        if (name in this[ASYNC_RESOLVED]) {
          var value = this[ASYNC_RESOLVED][name];
          return value || this[ASYNC_DEFAULTS][name];
        }

        if (name in this[ASYNC_ORIGINAL]) {
          var state = this[COMPONENT] && this[COMPONENT].internalState;

          if (state && state.hasAsyncProp(name)) {
            return state.getAsyncProp(name) || this[ASYNC_DEFAULTS][name];
          }
        }
      }

      return this[ASYNC_DEFAULTS][name];
    }
  };
}

function hasOwnProperty(object, prop) {
  return Object.prototype.hasOwnProperty.call(object, prop);
}

function getOwnProperty(object, prop) {
  return hasOwnProperty(object, prop) && object[prop];
}

function getComponentName(componentClass) {
  var componentName = getOwnProperty(componentClass, 'layerName') || getOwnProperty(componentClass, 'componentName');

  if (!componentName) {
    log.once(0, "".concat(componentClass.name, ".componentName not specified"))();
  }

  return componentName || componentClass.name;
}
//# sourceMappingURL=create-props.js.map