import { toHilbertQuadkey, FromHilbertQuadKey, IJToST, STToUV, FaceUVToXYZ, XYZToLngLat } from './s2-geometry';
import Long from 'long';

function getIdFromToken(token) {
  const paddedToken = token.padEnd(16, '0');
  return Long.fromString(paddedToken, 16);
}

const MAX_RESOLUTION = 100;

function getGeoBounds({
  face,
  ij,
  level
}) {
  const offsets = [[0, 0], [0, 1], [1, 1], [1, 0], [0, 0]];
  const resolution = Math.max(1, Math.ceil(MAX_RESOLUTION * Math.pow(2, -level)));
  const result = new Float64Array(4 * resolution * 2 + 2);
  let ptIndex = 0;

  for (let i = 0; i < 4; i++) {
    const offset = offsets[i].slice(0);
    const nextOffset = offsets[i + 1];
    const stepI = (nextOffset[0] - offset[0]) / resolution;
    const stepJ = (nextOffset[1] - offset[1]) / resolution;

    for (let j = 0; j < resolution; j++) {
      offset[0] += stepI;
      offset[1] += stepJ;
      const st = IJToST(ij, level, offset);
      const uv = STToUV(st);
      const xyz = FaceUVToXYZ(face, uv);
      const lngLat = XYZToLngLat(xyz);
      result[ptIndex++] = lngLat[0];
      result[ptIndex++] = lngLat[1];
    }
  }

  result[ptIndex++] = result[0];
  result[ptIndex++] = result[1];
  return result;
}

export function getS2QuadKey(token) {
  if (typeof token === 'string') {
    if (token.indexOf('/') > 0) {
      return token;
    }

    token = getIdFromToken(token);
  }

  return toHilbertQuadkey(token.toString());
}
export function getS2Polygon(token) {
  const key = getS2QuadKey(token);
  const s2cell = FromHilbertQuadKey(key);
  return getGeoBounds(s2cell);
}
//# sourceMappingURL=s2-utils.js.map