import Long from 'long';
const FACE_BITS = 3;
const MAX_LEVEL = 30;
const POS_BITS = 2 * MAX_LEVEL + 1;
const RADIAN_TO_DEGREE = 180 / Math.PI;
export function IJToST(ij, order, offsets) {
  const maxSize = 1 << order;
  return [(ij[0] + offsets[0]) / maxSize, (ij[1] + offsets[1]) / maxSize];
}

function singleSTtoUV(st) {
  if (st >= 0.5) {
    return 1 / 3.0 * (4 * st * st - 1);
  }

  return 1 / 3.0 * (1 - 4 * (1 - st) * (1 - st));
}

export function STToUV(st) {
  return [singleSTtoUV(st[0]), singleSTtoUV(st[1])];
}
export function FaceUVToXYZ(face, [u, v]) {
  switch (face) {
    case 0:
      return [1, u, v];

    case 1:
      return [-u, 1, v];

    case 2:
      return [-u, -v, 1];

    case 3:
      return [-1, -v, -u];

    case 4:
      return [v, -1, -u];

    case 5:
      return [v, u, -1];

    default:
      throw new Error('Invalid face');
  }
}
export function XYZToLngLat([x, y, z]) {
  const lat = Math.atan2(z, Math.sqrt(x * x + y * y));
  const lng = Math.atan2(y, x);
  return [lng * RADIAN_TO_DEGREE, lat * RADIAN_TO_DEGREE];
}
export function toHilbertQuadkey(idS) {
  let bin = Long.fromString(idS, true, 10).toString(2);

  while (bin.length < FACE_BITS + POS_BITS) {
    bin = '0' + bin;
  }

  const lsbIndex = bin.lastIndexOf('1');
  const faceB = bin.substring(0, 3);
  const posB = bin.substring(3, lsbIndex);
  const levelN = posB.length / 2;
  const faceS = Long.fromString(faceB, true, 2).toString(10);
  let posS = Long.fromString(posB, true, 2).toString(4);

  while (posS.length < levelN) {
    posS = '0' + posS;
  }

  return "".concat(faceS, "/").concat(posS);
}

function rotateAndFlipQuadrant(n, point, rx, ry) {
  if (ry === 0) {
    if (rx === 1) {
      point[0] = n - 1 - point[0];
      point[1] = n - 1 - point[1];
    }

    const x = point[0];
    point[0] = point[1];
    point[1] = x;
  }
}

export function FromHilbertQuadKey(hilbertQuadkey) {
  const parts = hilbertQuadkey.split('/');
  const face = parseInt(parts[0], 10);
  const position = parts[1];
  const maxLevel = position.length;
  const point = [0, 0];
  let level;

  for (let i = maxLevel - 1; i >= 0; i--) {
    level = maxLevel - i;
    const bit = position[i];
    let rx = 0;
    let ry = 0;

    if (bit === '1') {
      ry = 1;
    } else if (bit === '2') {
      rx = 1;
      ry = 1;
    } else if (bit === '3') {
      rx = 1;
    }

    const val = Math.pow(2, level - 1);
    rotateAndFlipQuadrant(val, point, rx, ry);
    point[0] += val * rx;
    point[1] += val * ry;
  }

  if (face % 2 === 1) {
    const t = point[0];
    point[0] = point[1];
    point[1] = t;
  }

  return {
    face,
    ij: point,
    level
  };
}
//# sourceMappingURL=s2-geometry.js.map