import { Deck } from '@deck.gl/core';
const MAX_LATITUDE = 85.05113;
export function createDeckInstance(map, overlay, deck) {
  if (deck) {
    if (deck.props.userData._googleMap === map) {
      return deck;
    }

    destroyDeckInstance(deck);
  }

  const eventListeners = {
    click: null,
    mousemove: null,
    mouseout: null
  };
  deck = new Deck({
    canvas: createDeckCanvas(overlay),
    initialViewState: {
      longitude: 0,
      latitude: 0,
      zoom: 1
    },
    controller: false,
    userData: {
      _googleMap: map,
      _eventListeners: eventListeners
    }
  });

  for (const eventType in eventListeners) {
    eventListeners[eventType] = map.addListener(eventType, evt => handleMouseEvent(deck, eventType, evt));
  }

  return deck;
}

function createDeckCanvas(overlay) {
  const container = overlay.getPanes().overlayLayer;
  const deckCanvas = document.createElement('canvas');
  Object.assign(deckCanvas.style, {
    position: 'absolute'
  });
  container.appendChild(deckCanvas);
  return deckCanvas;
}

export function destroyDeckInstance(deck) {
  const {
    _eventListeners: eventListeners
  } = deck.props.userData;

  for (const eventType in eventListeners) {
    eventListeners[eventType].remove();
  }

  deck.finalize();
  deck.canvas.parentNode.removeChild(deck.canvas);
}
export function getViewState(map, overlay) {
  const container = map.getDiv().firstChild;
  const width = container.offsetWidth;
  const height = container.offsetHeight;
  const projection = overlay.getProjection();
  const bounds = map.getBounds();
  const ne = bounds.getNorthEast();
  const sw = bounds.getSouthWest();
  const topRight = projection.fromLatLngToDivPixel(ne);
  const bottomLeft = projection.fromLatLngToDivPixel(sw);
  const nwContainerPx = new google.maps.Point(0, 0);
  const nw = projection.fromContainerPixelToLatLng(nwContainerPx);
  const nwDivPx = projection.fromLatLngToDivPixel(nw);
  let leftOffset = nwDivPx.x;
  let topOffset = nwDivPx.y;
  const mapWidth = projection.getWorldWidth();
  const mapCount = Math.ceil(width / mapWidth);
  leftOffset -= Math.floor(mapCount / 2) * mapWidth;
  const scale = height ? (bottomLeft.y - topRight.y) / height : 1;
  const zoom = Math.log2(scale || 1) + map.getZoom() - 1;
  let centerPx = new google.maps.Point(width / 2, height / 2);
  const centerContainer = projection.fromContainerPixelToLatLng(centerPx);
  let latitude = centerContainer.lat();
  const longitude = centerContainer.lng();

  if (Math.abs(latitude) > MAX_LATITUDE) {
    latitude = latitude > 0 ? MAX_LATITUDE : -MAX_LATITUDE;
    const center = new google.maps.LatLng(latitude, longitude);
    centerPx = projection.fromLatLngToContainerPixel(center);
    topOffset += centerPx.y - height / 2;
  }

  return {
    width,
    height,
    left: leftOffset,
    top: topOffset,
    zoom,
    pitch: map.getTilt(),
    latitude,
    longitude
  };
}

function handleMouseEvent(deck, type, event) {
  let callback;

  switch (type) {
    case 'click':
      deck._lastPointerDownInfo = deck.pickObject({
        x: event.pixel.x,
        y: event.pixel.y
      });
      callback = deck._onEvent;
      break;

    case 'mousemove':
      type = 'pointermove';
      callback = deck._onPointerMove;
      break;

    case 'mouseout':
      type = 'pointerleave';
      callback = deck._onPointerMove;
      break;

    default:
      return;
  }

  callback({
    type,
    offsetCenter: event.pixel,
    srcEvent: event
  });
}
//# sourceMappingURL=utils.js.map