"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nextPowOfTwo = nextPowOfTwo;
exports.buildMapping = buildMapping;
exports.autoWrapping = autoWrapping;
exports.transformParagraph = transformParagraph;
exports.getTextFromBuffer = getTextFromBuffer;

var _core = require("@deck.gl/core");

var MISSING_CHAR_WIDTH = 32;

function nextPowOfTwo(number) {
  return Math.pow(2, Math.ceil(Math.log2(number)));
}

function buildMapping(_ref) {
  var characterSet = _ref.characterSet,
      getFontWidth = _ref.getFontWidth,
      fontHeight = _ref.fontHeight,
      buffer = _ref.buffer,
      maxCanvasWidth = _ref.maxCanvasWidth,
      _ref$mapping = _ref.mapping,
      mapping = _ref$mapping === void 0 ? {} : _ref$mapping,
      _ref$xOffset = _ref.xOffset,
      xOffset = _ref$xOffset === void 0 ? 0 : _ref$xOffset,
      _ref$yOffset = _ref.yOffset,
      yOffset = _ref$yOffset === void 0 ? 0 : _ref$yOffset;
  var row = 0;
  var x = xOffset;
  var i = 0;
  var _iteratorNormalCompletion = true;
  var _didIteratorError = false;
  var _iteratorError = undefined;

  try {
    for (var _iterator = characterSet[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
      var _char = _step.value;

      if (!mapping[_char]) {
        var width = getFontWidth(_char, i++);

        if (x + width + buffer * 2 > maxCanvasWidth) {
          x = 0;
          row++;
        }

        mapping[_char] = {
          x: x + buffer,
          y: yOffset + row * (fontHeight + buffer * 2) + buffer,
          width: width,
          height: fontHeight
        };
        x += width + buffer * 2;
      }
    }
  } catch (err) {
    _didIteratorError = true;
    _iteratorError = err;
  } finally {
    try {
      if (!_iteratorNormalCompletion && _iterator["return"] != null) {
        _iterator["return"]();
      }
    } finally {
      if (_didIteratorError) {
        throw _iteratorError;
      }
    }
  }

  var rowHeight = fontHeight + buffer * 2;
  return {
    mapping: mapping,
    xOffset: x,
    yOffset: yOffset + row * rowHeight,
    canvasHeight: nextPowOfTwo(yOffset + (row + 1) * rowHeight)
  };
}

function getTextWidth(text, mapping) {
  var width = 0;

  for (var i = 0; i < text.length; i++) {
    var character = text[i];
    var frameWidth = null;
    var frame = mapping && mapping[character];

    if (frame) {
      frameWidth = frame.width;
    }

    width += frameWidth;
  }

  return width;
}

function breakAll(text, maxWidth, iconMapping) {
  var rows = [];
  var rowStartCharIndex = 0;
  var rowOffsetLeft = 0;

  for (var i = 0; i < text.length; i++) {
    var textWidth = getTextWidth(text[i], iconMapping);

    if (rowOffsetLeft + textWidth > maxWidth) {
      if (rowStartCharIndex < i) {
        rows.push(text.substring(rowStartCharIndex, i));
      }

      rowStartCharIndex = i;
      rowOffsetLeft = 0;
    }

    rowOffsetLeft += textWidth;
  }

  if (rowStartCharIndex < text.length) {
    rows.push(text.substring(rowStartCharIndex));
  }

  return {
    rows: rows,
    lastRowStartCharIndex: rowStartCharIndex,
    lastRowOffsetLeft: rowOffsetLeft
  };
}

function breakWord(text, maxWidth, iconMapping) {
  var rows = [];
  var rowStartCharIndex = 0;
  var groupStartCharIndex = 0;
  var rowOffsetLeft = 0;
  var group = null;

  for (var i = 0; i < text.length; i++) {
    if (text[i] === ' ') {
      group = text[i];
      groupStartCharIndex = i + 1;
    } else if (i + 1 < text.length && text[i + 1] === ' ' || i + 1 === text.length) {
      group = text.substring(groupStartCharIndex, i + 1);
      groupStartCharIndex = i + 1;
    } else {
      group = null;
    }

    if (group) {
      var groupWidth = getTextWidth(group, iconMapping);

      if (rowOffsetLeft + groupWidth > maxWidth) {
        var lastGroupStartIndex = groupStartCharIndex - group.length;

        if (rowStartCharIndex < lastGroupStartIndex) {
          rows.push(text.substring(rowStartCharIndex, lastGroupStartIndex));
          rowStartCharIndex = lastGroupStartIndex;
          rowOffsetLeft = 0;
        }

        if (groupWidth > maxWidth) {
          var subGroups = breakAll(group, maxWidth, iconMapping);

          if (subGroups.rows.length > 1) {
            rows = rows.concat(subGroups.rows.slice(0, subGroups.rows.length - 1));
          }

          rowStartCharIndex = rowStartCharIndex + subGroups.lastRowStartCharIndex;
          groupWidth = subGroups.lastRowOffsetLeft;
        }
      }

      rowOffsetLeft += groupWidth;
    }
  }

  if (rowStartCharIndex < text.length) {
    rows.push(text.substring(rowStartCharIndex));
  }

  return {
    rows: rows,
    lastRowStartCharIndex: rowStartCharIndex,
    lastRowOffsetLeft: rowOffsetLeft
  };
}

function autoWrapping(text, wordBreak, maxWidth, iconMapping) {
  if (wordBreak === 'break-all') {
    return breakAll(text, maxWidth, iconMapping);
  }

  return breakWord(text, maxWidth, iconMapping);
}

function transformRow(row, iconMapping) {
  var x = 0;
  var rowHeight = 0;
  var leftOffsets = new Array(row.length);

  for (var i = 0; i < row.length; i++) {
    var character = row[i];
    var frame = iconMapping[character];

    if (frame) {
      if (!rowHeight) {
        rowHeight = frame.height;
      }

      leftOffsets[i] = x + frame.width / 2;
      x += frame.width;
    } else {
      _core.log.warn("Missing character: ".concat(character))();

      leftOffsets[i] = x;
      x += MISSING_CHAR_WIDTH;
    }
  }

  return {
    leftOffsets: leftOffsets,
    rowWidth: x,
    rowHeight: rowHeight
  };
}

function transformParagraph(paragraph, lineHeight, wordBreak, maxWidth, iconMapping) {
  var result = new Array(paragraph.length);
  var autoWrappingEnabled = (wordBreak === 'break-word' || wordBreak === 'break-all') && isFinite(maxWidth) && maxWidth > 0;
  var size = [0, 0];
  var rowOffsetTop = 0;
  var lineStartIndex = 0;

  for (var i = 0; i <= paragraph.length; i++) {
    var _char2 = paragraph[i];
    var line = void 0;

    if (_char2 === '\n' || _char2 === undefined) {
      line = paragraph.slice(lineStartIndex, i);
    }

    if (line) {
      var rows = autoWrappingEnabled ? autoWrapping(line, wordBreak, maxWidth, iconMapping).rows : [line];
      var _iteratorNormalCompletion2 = true;
      var _didIteratorError2 = false;
      var _iteratorError2 = undefined;

      try {
        for (var _iterator2 = rows[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
          var row = _step2.value;

          var _transformRow = transformRow(row, iconMapping, lineHeight),
              rowWidth = _transformRow.rowWidth,
              rowHeight = _transformRow.rowHeight,
              leftOffsets = _transformRow.leftOffsets;

          var _iteratorNormalCompletion3 = true;
          var _didIteratorError3 = false;
          var _iteratorError3 = undefined;

          try {
            for (var _iterator3 = leftOffsets[Symbol.iterator](), _step3; !(_iteratorNormalCompletion3 = (_step3 = _iterator3.next()).done); _iteratorNormalCompletion3 = true) {
              var x = _step3.value;
              result[lineStartIndex++] = {
                x: x,
                y: rowOffsetTop + rowHeight / 2,
                rowWidth: rowWidth
              };
            }
          } catch (err) {
            _didIteratorError3 = true;
            _iteratorError3 = err;
          } finally {
            try {
              if (!_iteratorNormalCompletion3 && _iterator3["return"] != null) {
                _iterator3["return"]();
              }
            } finally {
              if (_didIteratorError3) {
                throw _iteratorError3;
              }
            }
          }

          rowOffsetTop = rowOffsetTop + rowHeight * lineHeight;
          size[0] = autoWrappingEnabled ? maxWidth : Math.max(size[0], rowWidth);
        }
      } catch (err) {
        _didIteratorError2 = true;
        _iteratorError2 = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion2 && _iterator2["return"] != null) {
            _iterator2["return"]();
          }
        } finally {
          if (_didIteratorError2) {
            throw _iteratorError2;
          }
        }
      }
    }

    if (_char2 === '\n') {
      result[lineStartIndex++] = {
        x: 0,
        y: 0,
        rowWidth: 0
      };
    }
  }

  size[1] = rowOffsetTop;
  return {
    characters: result,
    size: size
  };
}

function getTextFromBuffer(_ref2) {
  var value = _ref2.value,
      length = _ref2.length,
      stride = _ref2.stride,
      offset = _ref2.offset,
      startIndices = _ref2.startIndices;
  var bytesPerElement = value.BYTES_PER_ELEMENT;
  var elementStride = stride ? stride / bytesPerElement : 1;
  var elementOffset = offset ? offset / bytesPerElement : 0;
  var characterCount = startIndices[length] || Math.floor((value.length - elementOffset - bytesPerElement) / elementStride) + 1;
  var texts = new Array(length);
  var codes = value;

  if (elementStride > 1 || elementOffset > 0) {
    codes = new value.constructor(characterCount);

    for (var i = 0; i < characterCount; i++) {
      codes[i] = value[i * elementStride + elementOffset];
    }
  }

  for (var index = 0; index < length; index++) {
    var startIndex = startIndices[index];
    var endIndex = startIndices[index + 1] || characterCount;
    texts[index] = String.fromCodePoint.apply(null, codes.subarray(startIndex, endIndex));
  }

  return {
    texts: texts,
    characterCount: characterCount
  };
}
//# sourceMappingURL=utils.js.map