import { Layer, project32, gouraudLighting, picking } from '@deck.gl/core';
import { Model, Geometry } from '@luma.gl/core';
import vs from './point-cloud-layer-vertex.glsl';
import fs from './point-cloud-layer-fragment.glsl';
const DEFAULT_COLOR = [0, 0, 0, 255];
const DEFAULT_NORMAL = [0, 0, 1];
const defaultProps = {
  sizeUnits: 'pixels',
  pointSize: {
    type: 'number',
    min: 0,
    value: 10
  },
  getPosition: {
    type: 'accessor',
    value: x => x.position
  },
  getNormal: {
    type: 'accessor',
    value: DEFAULT_NORMAL
  },
  getColor: {
    type: 'accessor',
    value: DEFAULT_COLOR
  },
  material: true,
  radiusPixels: {
    deprecatedFor: 'pointSize'
  }
};

function normalizeData(data) {
  const {
    header,
    attributes
  } = data;

  if (!header || !attributes) {
    return;
  }

  data.length = header.vertexCount;

  if (attributes.POSITION) {
    attributes.instancePositions = attributes.POSITION;
  }

  if (attributes.NORMAL) {
    attributes.instanceNormals = attributes.NORMAL;
  }

  if (attributes.COLOR_0) {
    attributes.instanceColors = attributes.COLOR_0;
  }
}

export default class PointCloudLayer extends Layer {
  getShaders(id) {
    return super.getShaders({
      vs,
      fs,
      modules: [project32, gouraudLighting, picking]
    });
  }

  initializeState() {
    this.getAttributeManager().addInstanced({
      instancePositions: {
        size: 3,
        type: 5130,
        fp64: this.use64bitPositions(),
        transition: true,
        accessor: 'getPosition'
      },
      instanceNormals: {
        size: 3,
        transition: true,
        accessor: 'getNormal',
        defaultValue: DEFAULT_NORMAL
      },
      instanceColors: {
        size: this.props.colorFormat.length,
        type: 5121,
        normalized: true,
        transition: true,
        accessor: 'getColor',
        defaultValue: DEFAULT_COLOR
      }
    });
  }

  updateState({
    props,
    oldProps,
    changeFlags
  }) {
    super.updateState({
      props,
      oldProps,
      changeFlags
    });

    if (changeFlags.extensionsChanged) {
      const {
        gl
      } = this.context;

      if (this.state.model) {
        this.state.model.delete();
      }

      this.setState({
        model: this._getModel(gl)
      });
      this.getAttributeManager().invalidateAll();
    }

    if (changeFlags.dataChanged) {
      normalizeData(props.data);
    }
  }

  draw({
    uniforms
  }) {
    const {
      viewport
    } = this.context;
    const {
      pointSize,
      sizeUnits
    } = this.props;
    const sizeMultiplier = sizeUnits === 'meters' ? 1 / viewport.metersPerPixel : 1;
    this.state.model.setUniforms(Object.assign({}, uniforms, {
      radiusPixels: pointSize * sizeMultiplier
    })).draw();
  }

  _getModel(gl) {
    const positions = [];

    for (let i = 0; i < 3; i++) {
      const angle = i / 3 * Math.PI * 2;
      positions.push(Math.cos(angle) * 2, Math.sin(angle) * 2, 0);
    }

    return new Model(gl, Object.assign({}, this.getShaders(), {
      id: this.props.id,
      geometry: new Geometry({
        drawMode: 4,
        attributes: {
          positions: new Float32Array(positions)
        }
      }),
      isInstanced: true
    }));
  }

}
PointCloudLayer.layerName = 'PointCloudLayer';
PointCloudLayer.defaultProps = defaultProps;
//# sourceMappingURL=point-cloud-layer.js.map