"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDeckInstance = getDeckInstance;
exports.addLayer = addLayer;
exports.removeLayer = removeLayer;
exports.updateLayer = updateLayer;
exports.drawLayer = drawLayer;

var _slicedToArray2 = _interopRequireDefault(require("@babel/runtime/helpers/slicedToArray"));

var _core = require("@deck.gl/core");

function getDeckInstance(_ref) {
  var map = _ref.map,
      gl = _ref.gl,
      deck = _ref.deck;

  if (map.__deck) {
    return map.__deck;
  }

  var customRender = deck && deck.props._customRender;
  var deckProps = {
    useDevicePixels: true,
    _customRender: function _customRender() {
      map.triggerRepaint();

      if (customRender) {
        customRender();
      }
    },
    parameters: {
      depthMask: true,
      depthTest: true,
      blendFunc: [770, 771, 1, 771],
      blendEquation: 32774
    },
    userData: {
      isExternal: false,
      mapboxLayers: new Set()
    }
  };

  if (deck) {
    deck.setProps(deckProps);
    deck.props.userData.isExternal = true;
  } else {
    Object.assign(deckProps, {
      gl: gl,
      width: false,
      height: false,
      viewState: getViewState(map)
    });
    deck = new _core.Deck(deckProps);
    map.on('move', function () {
      return onMapMove(deck, map);
    });
    map.on('remove', function () {
      deck.finalize();
      map.__deck = null;
    });
  }

  deck.props.userData.mapboxVersion = getMapboxVersion(map);
  map.__deck = deck;
  map.on('render', function () {
    return afterRender(deck, map);
  });
  return deck;
}

function addLayer(deck, layer) {
  deck.props.userData.mapboxLayers.add(layer);
  updateLayers(deck);
}

function removeLayer(deck, layer) {
  deck.props.userData.mapboxLayers["delete"](layer);
  updateLayers(deck);
}

function updateLayer(deck, layer) {
  updateLayers(deck);
}

function drawLayer(deck, map, layer) {
  var currentViewport = deck.props.userData.currentViewport;

  if (!currentViewport) {
    currentViewport = getViewport(deck, map, true);
    deck.props.userData.currentViewport = currentViewport;
  }

  deck._drawLayers('mapbox-repaint', {
    viewports: [currentViewport],
    layers: getLayers(deck, function (deckLayer) {
      return shouldDrawLayer(layer.id, deckLayer);
    }),
    clearCanvas: false
  });
}

function getViewState(map) {
  var _map$getCenter = map.getCenter(),
      lng = _map$getCenter.lng,
      lat = _map$getCenter.lat;

  return {
    longitude: lng,
    latitude: lat,
    zoom: map.getZoom(),
    bearing: map.getBearing(),
    pitch: map.getPitch()
  };
}

function getMapboxVersion(map) {
  var major = 0;
  var minor = 0;

  if (map.version) {
    var _map$version$split$sl = map.version.split('.').slice(0, 2).map(Number);

    var _map$version$split$sl2 = (0, _slicedToArray2["default"])(_map$version$split$sl, 2);

    major = _map$version$split$sl2[0];
    minor = _map$version$split$sl2[1];
  }

  return {
    major: major,
    minor: minor
  };
}

function getViewport(deck, map) {
  var useMapboxProjection = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : true;
  var mapboxVersion = deck.props.userData.mapboxVersion;
  return new _core.WebMercatorViewport(Object.assign({
    x: 0,
    y: 0,
    width: deck.width,
    height: deck.height,
    repeat: true
  }, getViewState(map), useMapboxProjection ? {
    nearZMultiplier: mapboxVersion.major === 1 && mapboxVersion.minor >= 3 || mapboxVersion.major >= 2 ? 0.02 : 1 / (deck.height || 1)
  } : {
    nearZMultiplier: 0.1
  }));
}

function afterRender(deck, map) {
  var _deck$props$userData = deck.props.userData,
      mapboxLayers = _deck$props$userData.mapboxLayers,
      isExternal = _deck$props$userData.isExternal;

  if (isExternal) {
    var mapboxLayerIds = Array.from(mapboxLayers, function (layer) {
      return layer.id;
    });
    var layers = getLayers(deck, function (deckLayer) {
      var _iteratorNormalCompletion = true;
      var _didIteratorError = false;
      var _iteratorError = undefined;

      try {
        for (var _iterator = mapboxLayerIds[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
          var id = _step.value;

          if (shouldDrawLayer(id, deckLayer)) {
            return false;
          }
        }
      } catch (err) {
        _didIteratorError = true;
        _iteratorError = err;
      } finally {
        try {
          if (!_iteratorNormalCompletion && _iterator["return"] != null) {
            _iterator["return"]();
          }
        } finally {
          if (_didIteratorError) {
            throw _iteratorError;
          }
        }
      }

      return true;
    });

    if (layers.length > 0) {
      deck._drawLayers('mapbox-repaint', {
        viewports: [getViewport(deck, map, false)],
        layers: layers,
        clearCanvas: false
      });
    }
  }

  deck.props.userData.currentViewport = null;
}

function onMapMove(deck, map) {
  deck.setProps({
    viewState: getViewState(map)
  });
  deck.needsRedraw({
    clearRedrawFlags: true
  });
}

function getLayers(deck, layerFilter) {
  var layers = deck.layerManager.getLayers();
  return layers.filter(layerFilter);
}

function shouldDrawLayer(id, layer) {
  var layerInstance = layer;

  while (layerInstance) {
    if (layerInstance.id === id) {
      return true;
    }

    layerInstance = layerInstance.parent;
  }

  return false;
}

function updateLayers(deck) {
  if (deck.props.userData.isExternal) {
    return;
  }

  var layers = [];
  deck.props.userData.mapboxLayers.forEach(function (deckLayer) {
    var LayerType = deckLayer.props.type;
    var layer = new LayerType(deckLayer.props);
    layers.push(layer);
  });
  deck.setProps({
    layers: layers
  });
}
//# sourceMappingURL=deck-utils.js.map