import { Deck, WebMercatorViewport } from '@deck.gl/core';
export function getDeckInstance({
  map,
  gl,
  deck
}) {
  if (map.__deck) {
    return map.__deck;
  }

  const customRender = deck && deck.props._customRender;
  const deckProps = {
    useDevicePixels: true,
    _customRender: () => {
      map.triggerRepaint();

      if (customRender) {
        customRender();
      }
    },
    parameters: {
      depthMask: true,
      depthTest: true,
      blendFunc: [770, 771, 1, 771],
      blendEquation: 32774
    },
    userData: {
      isExternal: false,
      mapboxLayers: new Set()
    }
  };

  if (deck) {
    deck.setProps(deckProps);
    deck.props.userData.isExternal = true;
  } else {
    Object.assign(deckProps, {
      gl,
      width: false,
      height: false,
      viewState: getViewState(map)
    });
    deck = new Deck(deckProps);
    map.on('move', () => onMapMove(deck, map));
    map.on('remove', () => {
      deck.finalize();
      map.__deck = null;
    });
  }

  deck.props.userData.mapboxVersion = getMapboxVersion(map);
  map.__deck = deck;
  map.on('render', () => afterRender(deck, map));
  return deck;
}
export function addLayer(deck, layer) {
  deck.props.userData.mapboxLayers.add(layer);
  updateLayers(deck);
}
export function removeLayer(deck, layer) {
  deck.props.userData.mapboxLayers.delete(layer);
  updateLayers(deck);
}
export function updateLayer(deck, layer) {
  updateLayers(deck);
}
export function drawLayer(deck, map, layer) {
  let {
    currentViewport
  } = deck.props.userData;

  if (!currentViewport) {
    currentViewport = getViewport(deck, map, true);
    deck.props.userData.currentViewport = currentViewport;
  }

  deck._drawLayers('mapbox-repaint', {
    viewports: [currentViewport],
    layers: getLayers(deck, deckLayer => shouldDrawLayer(layer.id, deckLayer)),
    clearCanvas: false
  });
}

function getViewState(map) {
  const {
    lng,
    lat
  } = map.getCenter();
  return {
    longitude: lng,
    latitude: lat,
    zoom: map.getZoom(),
    bearing: map.getBearing(),
    pitch: map.getPitch()
  };
}

function getMapboxVersion(map) {
  let major = 0;
  let minor = 0;

  if (map.version) {
    [major, minor] = map.version.split('.').slice(0, 2).map(Number);
  }

  return {
    major,
    minor
  };
}

function getViewport(deck, map, useMapboxProjection = true) {
  const {
    mapboxVersion
  } = deck.props.userData;
  return new WebMercatorViewport(Object.assign({
    x: 0,
    y: 0,
    width: deck.width,
    height: deck.height,
    repeat: true
  }, getViewState(map), useMapboxProjection ? {
    nearZMultiplier: mapboxVersion.major === 1 && mapboxVersion.minor >= 3 || mapboxVersion.major >= 2 ? 0.02 : 1 / (deck.height || 1)
  } : {
    nearZMultiplier: 0.1
  }));
}

function afterRender(deck, map) {
  const {
    mapboxLayers,
    isExternal
  } = deck.props.userData;

  if (isExternal) {
    const mapboxLayerIds = Array.from(mapboxLayers, layer => layer.id);
    const layers = getLayers(deck, deckLayer => {
      for (const id of mapboxLayerIds) {
        if (shouldDrawLayer(id, deckLayer)) {
          return false;
        }
      }

      return true;
    });

    if (layers.length > 0) {
      deck._drawLayers('mapbox-repaint', {
        viewports: [getViewport(deck, map, false)],
        layers,
        clearCanvas: false
      });
    }
  }

  deck.props.userData.currentViewport = null;
}

function onMapMove(deck, map) {
  deck.setProps({
    viewState: getViewState(map)
  });
  deck.needsRedraw({
    clearRedrawFlags: true
  });
}

function getLayers(deck, layerFilter) {
  const layers = deck.layerManager.getLayers();
  return layers.filter(layerFilter);
}

function shouldDrawLayer(id, layer) {
  let layerInstance = layer;

  while (layerInstance) {
    if (layerInstance.id === id) {
      return true;
    }

    layerInstance = layerInstance.parent;
  }

  return false;
}

function updateLayers(deck) {
  if (deck.props.userData.isExternal) {
    return;
  }

  const layers = [];
  deck.props.userData.mapboxLayers.forEach(deckLayer => {
    const LayerType = deckLayer.props.type;
    const layer = new LayerType(deckLayer.props);
    layers.push(layer);
  });
  deck.setProps({
    layers
  });
}
//# sourceMappingURL=deck-utils.js.map