import React, { createElement } from 'react';
import PropTypes from 'prop-types';
import { Deck, _memoize as memoize } from '@deck.gl/core';
import extractJSXLayers from './utils/extract-jsx-layers';
import positionChildrenUnderViews from './utils/position-children-under-views';
import extractStyles from './utils/extract-styles';
const propTypes = Deck.getPropTypes(PropTypes);
const defaultProps = Deck.defaultProps;
export default class DeckGL extends React.Component {
  constructor(props) {
    super(props);
    this.viewports = null;
    this.children = null;
    this._needsRedraw = null;
    this._containerRef = React.createRef();
    this._canvasRef = React.createRef();
    this.pickObject = this.pickObject.bind(this);
    this.pickMultipleObjects = this.pickMultipleObjects.bind(this);
    this.pickObjects = this.pickObjects.bind(this);
    this._extractJSXLayers = memoize(extractJSXLayers);
    this._positionChildrenUnderViews = memoize(positionChildrenUnderViews);
    this._extractStyles = memoize(extractStyles);
  }

  componentDidMount() {
    const DeckClass = this.props.Deck || Deck;
    this.deck = this.deck || new DeckClass(Object.assign({}, this.props, {
      parent: this._containerRef.current,
      canvas: this._canvasRef.current,
      style: null,
      width: '100%',
      height: '100%',
      _customRender: this._customRender.bind(this)
    }));

    this._updateFromProps(this.props);
  }

  shouldComponentUpdate(nextProps) {
    this._updateFromProps(nextProps);

    const childrenChanged = this.children !== this._parseJSX(nextProps).children;

    const viewsChanged = this.deck.viewManager && this.deck.viewManager.needsRedraw();
    return childrenChanged && !viewsChanged;
  }

  componentDidUpdate() {
    this._redrawDeck();
  }

  componentWillUnmount() {
    this.deck.finalize();
  }

  pickObject(opts) {
    return this.deck.pickObject(opts);
  }

  pickMultipleObjects(opts) {
    return this.deck.pickMultipleObjects(opts);
  }

  pickObjects(opts) {
    return this.deck.pickObjects(opts);
  }

  _redrawDeck() {
    if (this._needsRedraw) {
      this.deck._drawLayers(this._needsRedraw);

      this._needsRedraw = null;
    }
  }

  _customRender(redrawReason) {
    this._needsRedraw = redrawReason;
    const viewports = this.deck.viewManager.getViewports();

    if (viewports !== this.viewports) {
      this.forceUpdate();
    } else {
      this._redrawDeck();
    }
  }

  _parseJSX(props) {
    return this._extractJSXLayers({
      layers: props.layers,
      views: props.views,
      children: props.children
    });
  }

  _updateFromProps(props) {
    const {
      layers,
      views
    } = this._parseJSX(props);

    const deckProps = Object.assign({}, props, {
      style: null,
      width: '100%',
      height: '100%',
      layers,
      views
    });
    this.deck.setProps(deckProps);
  }

  render() {
    const {
      ContextProvider,
      width,
      height,
      style
    } = this.props;
    const {
      viewManager
    } = this.deck || {};
    this.viewports = viewManager && viewManager.getViewports();
    this.children = this._parseJSX(this.props).children;

    const children = this._positionChildrenUnderViews({
      children: this.children,
      viewports: this.viewports,
      deck: this.deck,
      ContextProvider
    });

    const {
      containerStyle,
      canvasStyle
    } = this._extractStyles({
      width,
      height,
      style
    });

    const canvas = createElement('canvas', {
      key: 'canvas',
      ref: this._canvasRef,
      style: canvasStyle
    });
    return createElement('div', {
      id: 'deckgl-wrapper',
      ref: this._containerRef,
      style: containerStyle
    }, [canvas, children]);
  }

}
DeckGL.propTypes = propTypes;
DeckGL.defaultProps = defaultProps;
//# sourceMappingURL=deckgl.js.map