import { createElement } from 'react';
import { View } from '@deck.gl/core';
import { inheritsFrom } from './inherits-from';
import evaluateChildren from './evaluate-children';
export default function positionChildrenUnderViews({
  children,
  viewports,
  deck,
  ContextProvider
}) {
  const {
    viewManager
  } = deck || {};

  if (!viewManager || !viewManager.views.length) {
    return [];
  }

  const views = {};
  const defaultViewId = viewManager.views[0].id;

  for (const child of children) {
    let viewId = defaultViewId;
    let viewChildren = child;

    if (inheritsFrom(child.type, View)) {
      viewId = child.props.id || defaultViewId;
      viewChildren = child.props.children;
    }

    const viewport = viewManager.getViewport(viewId);
    const viewState = viewManager.getViewState(viewId);

    if (viewport) {
      const {
        x,
        y,
        width,
        height
      } = viewport;
      viewChildren = evaluateChildren(viewChildren, {
        x,
        y,
        width,
        height,
        viewport,
        viewState
      });

      if (!views[viewId]) {
        views[viewId] = {
          viewport,
          children: []
        };
      }

      views[viewId].children.push(viewChildren);
    }
  }

  return Object.keys(views).map(viewId => {
    const {
      viewport,
      children: viewChildren
    } = views[viewId];
    const {
      x,
      y,
      width,
      height
    } = viewport;
    const style = {
      position: 'absolute',
      left: x,
      top: y,
      width,
      height
    };
    const key = "view-".concat(viewId);
    const viewElement = createElement('div', {
      key,
      id: key,
      style
    }, ...viewChildren);

    if (ContextProvider) {
      const contextValue = {
        viewport,
        container: deck.canvas.offsetParent,
        eventManager: deck.eventManager,
        onViewStateChange: params => {
          params.viewId = viewId;

          deck._onViewStateChange(params);
        }
      };
      return createElement(ContextProvider, {
        key,
        value: contextValue
      }, viewElement);
    }

    return viewElement;
  });
}
//# sourceMappingURL=position-children-under-views.js.map