"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const utils_1 = require("@docusaurus/utils");
const chalk = require("chalk");
const chokidar_1 = __importDefault(require("chokidar"));
const express_1 = __importDefault(require("express"));
const html_webpack_plugin_1 = __importDefault(require("html-webpack-plugin"));
const path_1 = __importDefault(require("path"));
const openBrowser_1 = __importDefault(require("react-dev-utils/openBrowser"));
const WebpackDevServerUtils_1 = require("react-dev-utils/WebpackDevServerUtils");
const errorOverlayMiddleware_1 = __importDefault(require("react-dev-utils/errorOverlayMiddleware"));
const evalSourceMapMiddleware_1 = __importDefault(require("react-dev-utils/evalSourceMapMiddleware"));
const webpack_1 = __importDefault(require("webpack"));
const webpack_dev_server_1 = __importDefault(require("webpack-dev-server"));
const webpack_merge_1 = __importDefault(require("webpack-merge"));
const HotModuleReplacementPlugin_1 = __importDefault(require("webpack/lib/HotModuleReplacementPlugin"));
const server_1 = require("../server");
const constants_1 = require("../constants");
const client_1 = __importDefault(require("../webpack/client"));
const utils_2 = require("../webpack/utils");
const choosePort_1 = __importDefault(require("../choosePort"));
function getHost(reqHost) {
    return reqHost || 'localhost';
}
async function getPort(reqPort, host) {
    const basePort = reqPort ? parseInt(reqPort, 10) : constants_1.DEFAULT_PORT;
    const port = await choosePort_1.default(host, basePort);
    return port;
}
async function start(siteDir, cliOptions = {}) {
    process.env.NODE_ENV = 'development';
    process.env.BABEL_ENV = 'development';
    console.log(chalk.blue('Starting the development server...'));
    // Process all related files as a prop.
    const props = await server_1.load(siteDir);
    // Reload files processing.
    const reload = () => {
        server_1.load(siteDir).catch((err) => {
            console.error(chalk.red(err.stack));
        });
    };
    const { siteConfig, plugins = [] } = props;
    const normalizeToSiteDir = (filepath) => {
        if (filepath && path_1.default.isAbsolute(filepath)) {
            return utils_1.posixPath(path_1.default.relative(siteDir, filepath));
        }
        return utils_1.posixPath(filepath);
    };
    const pluginPaths = []
        .concat(...plugins
        .map((plugin) => { var _a, _b; return (_b = (_a = plugin.getPathsToWatch) === null || _a === void 0 ? void 0 : _a.call(plugin)) !== null && _b !== void 0 ? _b : []; })
        .filter(Boolean))
        .map(normalizeToSiteDir);
    const fsWatcher = chokidar_1.default.watch([...pluginPaths, constants_1.CONFIG_FILE_NAME], {
        cwd: siteDir,
        ignoreInitial: true,
    });
    ['add', 'change', 'unlink', 'addDir', 'unlinkDir'].forEach((event) => fsWatcher.on(event, reload));
    const protocol = process.env.HTTPS === 'true' ? 'https' : 'http';
    const host = getHost(cliOptions.host);
    const port = await getPort(cliOptions.port, host);
    if (port === null) {
        process.exit();
    }
    const { baseUrl, headTags, preBodyTags, postBodyTags } = props;
    const urls = WebpackDevServerUtils_1.prepareUrls(protocol, host, port);
    const openUrl = utils_1.normalizeUrl([urls.localUrlForBrowser, baseUrl]);
    let config = webpack_merge_1.default(client_1.default(props), {
        plugins: [
            // Generates an `index.html` file with the <script> injected.
            new html_webpack_plugin_1.default({
                template: path_1.default.resolve(__dirname, '../client/templates/index.html.template.ejs'),
                // So we can define the position where the scripts are injected.
                inject: false,
                filename: 'index.html',
                title: siteConfig.title,
                headTags,
                preBodyTags,
                postBodyTags,
            }),
            // This is necessary to emit hot updates for webpack-dev-server.
            new HotModuleReplacementPlugin_1.default(),
        ],
    });
    // Plugin Lifecycle - configureWebpack.
    plugins.forEach((plugin) => {
        const { configureWebpack } = plugin;
        if (!configureWebpack) {
            return;
        }
        config = utils_2.applyConfigureWebpack(configureWebpack.bind(plugin), // The plugin lifecycle may reference `this`.
        config, false);
    });
    // https://webpack.js.org/configuration/dev-server
    const devServerConfig = Object.assign({
        compress: true,
        clientLogLevel: 'error',
        hot: true,
        hotOnly: cliOptions.hotOnly,
        // Use 'ws' instead of 'sockjs-node' on server since we're using native
        // websockets in `webpackHotDevClient`.
        transportMode: 'ws',
        // Prevent a WS client from getting injected as we're already including
        // `webpackHotDevClient`.
        injectClient: false,
        quiet: true,
        headers: {
            'access-control-allow-origin': '*',
        },
        publicPath: baseUrl,
        watchOptions: {
            ignored: /node_modules/,
            poll: cliOptions.poll,
        },
        historyApiFallback: {
            rewrites: [{ from: /\/*/, to: baseUrl }],
        },
        disableHostCheck: true,
        // Disable overlay on browser since we use CRA's overlay error reporting.
        overlay: false,
        host,
        before: (app, server) => {
            app.use(baseUrl, express_1.default.static(path_1.default.resolve(siteDir, constants_1.STATIC_DIR_NAME)));
            // This lets us fetch source contents from webpack for the error overlay.
            app.use(evalSourceMapMiddleware_1.default(server));
            // This lets us open files from the runtime error overlay.
            app.use(errorOverlayMiddleware_1.default());
            // TODO: add plugins beforeDevServer and afterDevServer hook
        },
    }, config.devServer);
    const compiler = webpack_1.default(config);
    if (process.env.E2E_TEST) {
        compiler.hooks.done.tap('done', (stats) => {
            if (stats.hasErrors()) {
                console.log('E2E_TEST: Project has compiler errors.');
                process.exit(1);
            }
            console.log('E2E_TEST: Project can compile.');
            process.exit(0);
        });
    }
    const devServer = new webpack_dev_server_1.default(compiler, devServerConfig);
    devServer.listen(port, host, (err) => {
        if (err) {
            console.log(err);
        }
        if (cliOptions.open) {
            openBrowser_1.default(openUrl);
        }
    });
    ['SIGINT', 'SIGTERM'].forEach((sig) => {
        process.on(sig, () => {
            devServer.close();
            process.exit();
        });
    });
}
exports.default = start;
