"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleBrokenLinks = exports.getBrokenLinksErrorMessage = exports.getAllBrokenLinks = void 0;
const react_router_config_1 = require("react-router-config");
const resolve_pathname_1 = __importDefault(require("resolve-pathname"));
const chalk_1 = __importDefault(require("chalk"));
const fs_extra_1 = __importDefault(require("fs-extra"));
const lodash_1 = require("lodash");
const utils_1 = require("@docusaurus/utils");
function toReactRouterRoutes(routes) {
    // @ts-expect-error: types incompatible???
    return routes;
}
// matchRoutes does not support qs/anchors, so we remove it!
function onlyPathname(link) {
    return link.split('#')[0].split('?')[0];
}
function getPageBrokenLinks({ pagePath, pageLinks, routes, }) {
    // ReactRouter is able to support links like ./../somePath
    // but matchRoutes does not do this resolving internally
    // we must resolve the links before using matchRoutes
    // resolvePathname is used internally by ReactRouter
    function resolveLink(link) {
        const resolvedLink = resolve_pathname_1.default(onlyPathname(link), pagePath);
        return { link, resolvedLink };
    }
    function isBrokenLink(link) {
        const matchedRoutes = react_router_config_1.matchRoutes(toReactRouterRoutes(routes), link);
        return matchedRoutes.length === 0;
    }
    return pageLinks.map(resolveLink).filter((l) => isBrokenLink(l.resolvedLink));
}
// The route defs can be recursive, and have a parent match-all route
// We don't want to match broken links like /docs/brokenLink against /docs/*
// For this reason, we only consider the "final routes", that do not have subroutes
// We also need to remove the match all 404 route
function filterIntermediateRoutes(routesInput) {
    function getFinalRoutes(route) {
        return route.routes ? lodash_1.flatMap(route.routes, getFinalRoutes) : [route];
    }
    const routesWithout404 = routesInput.filter((route) => route.path !== '*');
    return lodash_1.flatMap(routesWithout404, getFinalRoutes);
}
function getAllBrokenLinks({ allCollectedLinks, routes, }) {
    const filteredRoutes = filterIntermediateRoutes(routes);
    const allBrokenLinks = lodash_1.mapValues(allCollectedLinks, (pageLinks, pagePath) => {
        return getPageBrokenLinks({ pageLinks, pagePath, routes: filteredRoutes });
    });
    // remove pages without any broken link
    return lodash_1.pickBy(allBrokenLinks, (brokenLinks) => brokenLinks.length > 0);
}
exports.getAllBrokenLinks = getAllBrokenLinks;
function getBrokenLinksErrorMessage(allBrokenLinks) {
    if (Object.keys(allBrokenLinks).length === 0) {
        return undefined;
    }
    function brokenLinkMessage(brokenLink) {
        const showResolvedLink = brokenLink.link !== brokenLink.resolvedLink;
        return `${brokenLink.link}${showResolvedLink ? ` (resolved as: ${brokenLink.resolvedLink})` : ''}`;
    }
    function pageBrokenLinksMessage(pagePath, brokenLinks) {
        return `\n\n- Page path = ${pagePath}:\n   -> link to ${brokenLinks
            .map(brokenLinkMessage)
            .join('\n   -> link to ')}`;
    }
    return (`Broken links found!` +
        `${Object.entries(allBrokenLinks)
            .map(([pagePath, brokenLinks]) => pageBrokenLinksMessage(pagePath, brokenLinks))
            .join('\n')}
`);
}
exports.getBrokenLinksErrorMessage = getBrokenLinksErrorMessage;
// If a file actually exist on the file system, we know the link is valid
// even if docusaurus does not know about this file, so we don't report it
async function filterExistingFileLinks({ baseUrl, outDir, allCollectedLinks, }) {
    // not easy to make this async :'(
    function linkFileExists(link) {
        const filePath = `${outDir}/${utils_1.removePrefix(link, baseUrl)}`;
        try {
            return fs_extra_1.default.statSync(filePath).isFile(); // only consider files
        }
        catch (e) {
            return false;
        }
    }
    return lodash_1.mapValues(allCollectedLinks, (links) => {
        return links.filter((link) => !linkFileExists(link));
    });
}
async function handleBrokenLinks({ allCollectedLinks, onBrokenLinks, routes, baseUrl, outDir, }) {
    if (onBrokenLinks === 'ignore') {
        return;
    }
    // If we link to a file like /myFile.zip, and the file actually exist for the file system
    // it is not a broken link, it may simply be a link to an existing static file...
    const allCollectedLinksFiltered = await filterExistingFileLinks({
        allCollectedLinks,
        baseUrl,
        outDir,
    });
    const allBrokenLinks = getAllBrokenLinks({
        allCollectedLinks: allCollectedLinksFiltered,
        routes,
    });
    const errorMessage = getBrokenLinksErrorMessage(allBrokenLinks);
    if (errorMessage) {
        const finalMessage = `${errorMessage}\nNote: it's possible to ignore broken links with the 'onBrokenLinks' Docusaurus configuration.\n\n`;
        // Useful to ensure the CI fails in case of broken link
        if (onBrokenLinks === 'throw') {
            throw new Error(finalMessage);
        }
        else if (onBrokenLinks === 'error') {
            console.error(chalk_1.default.red(finalMessage));
        }
        else if (onBrokenLinks === 'log') {
            console.log(chalk_1.default.blue(finalMessage));
        }
        else {
            throw new Error(`unexpected onBrokenLinks value=${onBrokenLinks}`);
        }
    }
}
exports.handleBrokenLinks = handleBrokenLinks;
