"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadPlugins = exports.sortConfig = void 0;
const utils_1 = require("@docusaurus/utils");
const fs_extra_1 = __importDefault(require("fs-extra"));
const path_1 = __importDefault(require("path"));
const init_1 = __importDefault(require("./init"));
const DefaultPluginId = 'default';
function sortConfig(routeConfigs) {
    // Sort the route config. This ensures that route with nested
    // routes is always placed last.
    routeConfigs.sort((a, b) => {
        if (a.routes && !b.routes) {
            return 1;
        }
        if (!a.routes && b.routes) {
            return -1;
        }
        // Higher priority get placed first.
        if (a.priority || b.priority) {
            const priorityA = a.priority || 0;
            const priorityB = b.priority || 0;
            const score = priorityB - priorityA;
            if (score !== 0) {
                return score;
            }
        }
        return a.path.localeCompare(b.path);
    });
    routeConfigs.forEach((routeConfig) => {
        var _a;
        (_a = routeConfig.routes) === null || _a === void 0 ? void 0 : _a.sort((a, b) => a.path.localeCompare(b.path));
    });
}
exports.sortConfig = sortConfig;
async function loadPlugins({ pluginConfigs, context, }) {
    // 1. Plugin Lifecycle - Initialization/Constructor.
    const plugins = init_1.default({
        pluginConfigs,
        context,
    });
    // 2. Plugin Lifecycle - loadContent.
    // Currently plugins run lifecycle methods in parallel and are not order-dependent.
    // We could change this in future if there are plugins which need to
    // run in certain order or depend on others for data.
    const pluginsLoadedContent = await Promise.all(plugins.map(async (plugin) => {
        if (!plugin.loadContent) {
            return null;
        }
        return plugin.loadContent();
    }));
    // 3. Plugin Lifecycle - contentLoaded.
    const pluginsRouteConfigs = [];
    const globalData = {};
    await Promise.all(plugins.map(async (plugin, index) => {
        var _a;
        if (!plugin.contentLoaded) {
            return;
        }
        const pluginId = (_a = plugin.options.id) !== null && _a !== void 0 ? _a : DefaultPluginId;
        const pluginContentDir = path_1.default.join(context.generatedFilesDir, plugin.name);
        const addRoute = (config) => pluginsRouteConfigs.push(config);
        const createData = async (name, content) => {
            const modulePath = path_1.default.join(pluginContentDir, name);
            await fs_extra_1.default.ensureDir(path_1.default.dirname(modulePath));
            await utils_1.generate(pluginContentDir, name, content);
            return modulePath;
        };
        // the plugins global data are namespaced to avoid data conflicts:
        // - by plugin name
        // - by plugin id (allow using multiple instances of the same plugin)
        const setGlobalData = (data) => {
            var _a;
            globalData[plugin.name] = (_a = globalData[plugin.name]) !== null && _a !== void 0 ? _a : {};
            globalData[plugin.name][pluginId] = data;
        };
        const actions = {
            addRoute,
            createData,
            setGlobalData,
        };
        await plugin.contentLoaded({
            content: pluginsLoadedContent[index],
            actions,
        });
    }));
    // 4. Plugin Lifecycle - routesLoaded.
    // Currently plugins run lifecycle methods in parallel and are not order-dependent.
    // We could change this in future if there are plugins which need to
    // run in certain order or depend on others for data.
    await Promise.all(plugins.map(async (plugin) => {
        if (!plugin.routesLoaded) {
            return null;
        }
        return plugin.routesLoaded(pluginsRouteConfigs);
    }));
    // Sort the route config. This ensures that route with nested
    // routes are always placed last.
    sortConfig(pluginsRouteConfigs);
    return {
        plugins,
        pluginsRouteConfigs,
        globalData,
    };
}
exports.loadPlugins = loadPlugins;
