"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateConfig = exports.DEFAULT_CONFIG = void 0;
const constants_1 = require("../constants");
const joi_1 = __importDefault(require("@hapi/joi"));
const utils_validation_1 = require("@docusaurus/utils-validation");
exports.DEFAULT_CONFIG = {
    onBrokenLinks: 'throw',
    onDuplicateRoutes: 'warn',
    plugins: [],
    themes: [],
    presets: [],
    customFields: {},
    themeConfig: {},
};
const PluginSchema = joi_1.default.alternatives().try(joi_1.default.string(), joi_1.default.array().items(joi_1.default.string().required(), joi_1.default.object().required()).length(2));
const ThemeSchema = joi_1.default.alternatives().try(joi_1.default.string(), joi_1.default.array().items(joi_1.default.string().required(), joi_1.default.object().required()).length(2));
const PresetSchema = joi_1.default.alternatives().try(joi_1.default.string(), joi_1.default.array().items(joi_1.default.string().required(), joi_1.default.object().required()).length(2));
// TODO move to @docusaurus/utils-validation
const ConfigSchema = joi_1.default.object({
    baseUrl: joi_1.default.string()
        .required()
        .regex(new RegExp('/$', 'm'))
        .message('{{#label}} must be a string with a trailing `/`'),
    favicon: joi_1.default.string().required(),
    title: joi_1.default.string().required(),
    url: utils_validation_1.URISchema.required(),
    onBrokenLinks: joi_1.default.string()
        .equal('ignore', 'log', 'warn', 'error', 'throw')
        .default(exports.DEFAULT_CONFIG.onBrokenLinks),
    onDuplicateRoutes: joi_1.default.string()
        .equal('ignore', 'log', 'warn', 'error', 'throw')
        .default(exports.DEFAULT_CONFIG.onDuplicateRoutes),
    organizationName: joi_1.default.string().allow(''),
    projectName: joi_1.default.string().allow(''),
    customFields: joi_1.default.object().unknown().default(exports.DEFAULT_CONFIG.customFields),
    githubHost: joi_1.default.string(),
    plugins: joi_1.default.array().items(PluginSchema).default(exports.DEFAULT_CONFIG.plugins),
    themes: joi_1.default.array().items(ThemeSchema).default(exports.DEFAULT_CONFIG.themes),
    presets: joi_1.default.array().items(PresetSchema).default(exports.DEFAULT_CONFIG.presets),
    themeConfig: joi_1.default.object().unknown().default(exports.DEFAULT_CONFIG.themeConfig),
    scripts: joi_1.default.array().items(joi_1.default.string(), joi_1.default.object({
        src: joi_1.default.string().required(),
        async: joi_1.default.bool(),
        defer: joi_1.default.bool(),
    })
        // See https://github.com/facebook/docusaurus/issues/3378
        .unknown()),
    ssrTemplate: joi_1.default.string(),
    stylesheets: joi_1.default.array().items(joi_1.default.string(), joi_1.default.object({
        href: joi_1.default.string().required(),
        type: joi_1.default.string().required(),
    }).unknown()),
    tagline: joi_1.default.string().allow(''),
});
// TODO move to @docusaurus/utils-validation
function validateConfig(config) {
    const { error, value } = ConfigSchema.validate(config, {
        abortEarly: false,
    });
    if (error) {
        utils_validation_1.logValidationBugReportHint();
        if (utils_validation_1.isValidationDisabledEscapeHatch) {
            console.error(error);
            return config;
        }
        const unknownFields = error.details.reduce((formattedError, err) => {
            if (err.type === 'object.unknown') {
                return `${formattedError}"${err.path}",`;
            }
            return formattedError;
        }, '');
        let formattedError = error.details.reduce((accumulatedErr, err) => err.type !== 'object.unknown'
            ? `${accumulatedErr}${err.message}\n`
            : accumulatedErr, '');
        formattedError = unknownFields
            ? `${formattedError}These field(s) [${unknownFields}] are not recognized in ${constants_1.CONFIG_FILE_NAME}.\nIf you still want these fields to be in your configuration, put them in the 'customFields' attribute.\nSee https://v2.docusaurus.io/docs/docusaurus.config.js/#customfields`
            : formattedError;
        throw new Error(formattedError);
    }
    else {
        return value;
    }
}
exports.validateConfig = validateConfig;
