"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const module_1 = __importDefault(require("module"));
const path_1 = require("path");
const import_fresh_1 = __importDefault(require("import-fresh"));
const constants_1 = require("../../constants");
const versions_1 = require("../versions");
const pluginIds_1 = require("./pluginIds");
const utils_validation_1 = require("@docusaurus/utils-validation");
function initPlugins({ pluginConfigs, context, }) {
    // We need to resolve plugins from the perspective of the siteDir, since the siteDir's package.json
    // declares the dependency on these plugins.
    // We need to fallback to createRequireFromPath since createRequire is only available in node v12.
    // See: https://nodejs.org/api/modules.html#modules_module_createrequire_filename
    const createRequire = module_1.default.createRequire || module_1.default.createRequireFromPath;
    const pluginRequire = createRequire(path_1.join(context.siteDir, constants_1.CONFIG_FILE_NAME));
    const plugins = pluginConfigs
        .map((pluginItem) => {
        var _a, _b, _c, _d, _e;
        let pluginModuleImport;
        let pluginOptions = {};
        if (!pluginItem) {
            return null;
        }
        if (typeof pluginItem === 'string') {
            pluginModuleImport = pluginItem;
        }
        else if (Array.isArray(pluginItem)) {
            [pluginModuleImport, pluginOptions = {}] = pluginItem;
        }
        if (!pluginModuleImport) {
            return null;
        }
        // The pluginModuleImport value is any valid
        // module identifier - npm package or locally-resolved path.
        const pluginPath = pluginRequire.resolve(pluginModuleImport);
        const pluginModule = import_fresh_1.default(pluginPath);
        const pluginVersion = versions_1.getPluginVersion(pluginPath, context.siteDir);
        const plugin = pluginModule.default || pluginModule;
        // support both commonjs and ES modules
        const validateOptions = (_b = (_a = pluginModule.default) === null || _a === void 0 ? void 0 : _a.validateOptions) !== null && _b !== void 0 ? _b : pluginModule.validateOptions;
        if (validateOptions) {
            const normalizedOptions = validateOptions({
                validate: utils_validation_1.normalizePluginOptions,
                options: pluginOptions,
            });
            pluginOptions = normalizedOptions;
        }
        else {
            // Important to ensure all plugins have an id
            // as we don't go through the Joi schema that adds it
            pluginOptions = Object.assign(Object.assign({}, pluginOptions), { id: (_c = pluginOptions.id) !== null && _c !== void 0 ? _c : constants_1.DEFAULT_PLUGIN_ID });
        }
        // support both commonjs and ES modules
        const validateThemeConfig = (_e = (_d = pluginModule.default) === null || _d === void 0 ? void 0 : _d.validateThemeConfig) !== null && _e !== void 0 ? _e : pluginModule.validateThemeConfig;
        if (validateThemeConfig) {
            const normalizedThemeConfig = validateThemeConfig({
                validate: utils_validation_1.normalizeThemeConfig,
                themeConfig: context.siteConfig.themeConfig,
            });
            context.siteConfig.themeConfig = Object.assign(Object.assign({}, context.siteConfig.themeConfig), normalizedThemeConfig);
        }
        return Object.assign(Object.assign({}, plugin(context, pluginOptions)), { options: pluginOptions, version: pluginVersion });
    })
        .filter(Boolean);
    pluginIds_1.ensureUniquePluginInstanceIds(plugins);
    return plugins;
}
exports.default = initPlugins;
