"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = __importDefault(require("path"));
const del_1 = require("del");
class CleanWebpackPlugin {
    constructor(options = {}) {
        if (typeof options !== 'object' || Array.isArray(options) === true) {
            throw new Error(`clean-webpack-plugin only accepts an options object. See:
            https://github.com/johnagan/clean-webpack-plugin#options-and-defaults-optional`);
        }
        if (options.allowExternal) {
            throw new Error('clean-webpack-plugin: `allowExternal` option no longer supported. Use `dangerouslyAllowCleanPatternsOutsideProject`');
        }
        if (options.dangerouslyAllowCleanPatternsOutsideProject === true &&
            options.dry !== true &&
            options.dry !== false) {
            // eslint-disable-next-line no-console
            console.warn('clean-webpack-plugin: dangerouslyAllowCleanPatternsOutsideProject requires dry: false to be explicitly set. Enabling dry mode');
        }
        this.dangerouslyAllowCleanPatternsOutsideProject =
            options.dangerouslyAllowCleanPatternsOutsideProject === true || false;
        this.dry =
            options.dry === true || options.dry === false
                ? options.dry
                : this.dangerouslyAllowCleanPatternsOutsideProject === true || false;
        this.verbose = this.dry === true || options.verbose === true || false;
        this.cleanStaleWebpackAssets =
            options.cleanStaleWebpackAssets === true ||
                options.cleanStaleWebpackAssets === false
                ? options.cleanStaleWebpackAssets
                : true;
        this.protectWebpackAssets =
            options.protectWebpackAssets === true ||
                options.protectWebpackAssets === false
                ? options.protectWebpackAssets
                : true;
        this.cleanAfterEveryBuildPatterns = Array.isArray(options.cleanAfterEveryBuildPatterns)
            ? options.cleanAfterEveryBuildPatterns
            : [];
        this.cleanOnceBeforeBuildPatterns = Array.isArray(options.cleanOnceBeforeBuildPatterns)
            ? options.cleanOnceBeforeBuildPatterns
            : ['**/*'];
        /**
         * Store webpack build assets
         */
        this.currentAssets = [];
        /**
         * Only used with cleanOnceBeforeBuildPatterns
         */
        this.initialClean = false;
        this.outputPath = '';
        this.apply = this.apply.bind(this);
        this.handleInitial = this.handleInitial.bind(this);
        this.handleDone = this.handleDone.bind(this);
        this.removeFiles = this.removeFiles.bind(this);
    }
    apply(compiler) {
        if (!compiler.options.output || !compiler.options.output.path) {
            // eslint-disable-next-line no-console
            console.warn('clean-webpack-plugin: options.output.path not defined. Plugin disabled...');
            return;
        }
        this.outputPath = compiler.options.output.path;
        /**
         * webpack 4+ comes with a new plugin system.
         *
         * Check for hooks in-order to support old plugin system
         */
        const { hooks } = compiler;
        if (this.cleanOnceBeforeBuildPatterns.length !== 0) {
            if (hooks) {
                hooks.compile.tap('clean-webpack-plugin', () => {
                    this.handleInitial();
                });
            }
            else {
                compiler.plugin('compile', () => {
                    this.handleInitial();
                });
            }
        }
        if (hooks) {
            hooks.done.tap('clean-webpack-plugin', (stats) => {
                this.handleDone(stats);
            });
        }
        else {
            compiler.plugin('done', (stats) => {
                this.handleDone(stats);
            });
        }
    }
    /**
     * Initially remove files from output directory prior to build.
     *
     * Only happens once.
     *
     * Warning: It is recommended to initially clean your build directory outside of webpack to minimize unexpected behavior.
     */
    handleInitial() {
        if (this.initialClean) {
            return;
        }
        this.initialClean = true;
        this.removeFiles(this.cleanOnceBeforeBuildPatterns);
    }
    handleDone(stats) {
        /**
         * Do nothing if there is a webpack error
         */
        if (stats.hasErrors()) {
            if (this.verbose) {
                // eslint-disable-next-line no-console
                console.warn('clean-webpack-plugin: pausing due to webpack errors');
            }
            return;
        }
        /**
         * Fetch Webpack's output asset files
         */
        const statsAssets = stats.toJson({
            all: false,
            assets: true,
        }, true).assets || [];
        const assets = statsAssets.map((asset) => {
            return asset.name;
        });
        /**
         * Get all files that were in the previous build but not the current
         *
         * (relies on del's cwd: outputPath option)
         */
        const staleFiles = this.currentAssets.filter((previousAsset) => {
            const assetCurrent = assets.includes(previousAsset) === false;
            return assetCurrent;
        });
        /**
         * Save assets for next compilation
         */
        this.currentAssets = assets.sort();
        const removePatterns = [];
        /**
         * Remove unused webpack assets
         */
        if (this.cleanStaleWebpackAssets === true && staleFiles.length !== 0) {
            removePatterns.push(...staleFiles);
        }
        /**
         * Remove cleanAfterEveryBuildPatterns
         */
        if (this.cleanAfterEveryBuildPatterns.length !== 0) {
            removePatterns.push(...this.cleanAfterEveryBuildPatterns);
        }
        if (removePatterns.length !== 0) {
            this.removeFiles(removePatterns);
        }
    }
    removeFiles(patterns) {
        try {
            const deleted = del_1.sync(patterns, {
                force: this.dangerouslyAllowCleanPatternsOutsideProject,
                // Change context to build directory
                cwd: this.outputPath,
                dryRun: this.dry,
                dot: true,
                ignore: this.protectWebpackAssets ? this.currentAssets : [],
            });
            /**
             * Log if verbose is enabled
             */
            if (this.verbose) {
                deleted.forEach((file) => {
                    const filename = path_1.default.relative(process.cwd(), file);
                    const message = this.dry ? 'dry' : 'removed';
                    /**
                     * Use console.warn over .log
                     * https://github.com/webpack/webpack/issues/1904
                     * https://github.com/johnagan/clean-webpack-plugin/issues/11
                     */
                    // eslint-disable-next-line no-console
                    console.warn(`clean-webpack-plugin: ${message} ${filename}`);
                });
            }
        }
        catch (error) {
            const needsForce = /Cannot delete files\/folders outside the current working directory\./.test(error.message);
            if (needsForce) {
                const message = 'clean-webpack-plugin: Cannot delete files/folders outside the current working directory. Can be overridden with the `dangerouslyAllowCleanPatternsOutsideProject` option.';
                throw new Error(message);
            }
            throw error;
        }
    }
}
exports.default = CleanWebpackPlugin;
