"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDocVersionSuggestions = exports.getActiveDocContext = exports.getActiveVersion = exports.getLatestVersion = exports.getActivePlugin = void 0;
const router_1 = require("@docusaurus/router");
function getActivePlugin(allPluginDatas, pathname, options = {}) {
    const activeEntry = Object.entries(allPluginDatas).find(([_id, pluginData]) => {
        return !!router_1.matchPath(pathname, {
            path: pluginData.path,
            exact: false,
            strict: false,
        });
    });
    const activePlugin = activeEntry
        ? { pluginId: activeEntry[0], pluginData: activeEntry[1] }
        : undefined;
    if (!activePlugin && options.failfast) {
        throw new Error(`Can't find active docs plugin for pathname=${pathname}, while it was expected to be found. Maybe you tried to use a docs feature that can only be used on a docs-related page? Existing docs plugin paths are: ${Object.values(allPluginDatas)
            .map((plugin) => plugin.path)
            .join(', ')}`);
    }
    return activePlugin;
}
exports.getActivePlugin = getActivePlugin;
exports.getLatestVersion = (data) => {
    return data.versions.find((version) => version.isLast);
};
// Note: return undefined on doc-unrelated pages,
// because there's no version currently considered as active
exports.getActiveVersion = (data, pathname) => {
    const lastVersion = exports.getLatestVersion(data);
    // Last version is a route like /docs/*,
    // we need to try to match it last or it would match /docs/version-1.0/* as well
    const orderedVersionsMetadata = [
        ...data.versions.filter((version) => version !== lastVersion),
        lastVersion,
    ];
    return orderedVersionsMetadata.find((version) => {
        return !!router_1.matchPath(pathname, {
            path: version.path,
            exact: false,
            strict: false,
        });
    });
};
exports.getActiveDocContext = (data, pathname) => {
    const activeVersion = exports.getActiveVersion(data, pathname);
    const activeDoc = activeVersion === null || activeVersion === void 0 ? void 0 : activeVersion.docs.find((doc) => !!router_1.matchPath(pathname, {
        path: doc.path,
        exact: true,
        strict: false,
    }));
    function getAlternateVersionDocs(docId) {
        const result = {};
        data.versions.forEach((version) => {
            version.docs.forEach((doc) => {
                if (doc.id === docId) {
                    result[version.name] = doc;
                }
            });
        });
        return result;
    }
    const alternateVersionDocs = activeDoc
        ? getAlternateVersionDocs(activeDoc.id)
        : {};
    return {
        activeVersion,
        activeDoc,
        alternateDocVersions: alternateVersionDocs,
    };
};
exports.getDocVersionSuggestions = (data, pathname) => {
    const latestVersion = exports.getLatestVersion(data);
    const activeDocContext = exports.getActiveDocContext(data, pathname);
    // We only suggest another doc/version if user is not using the latest version
    const isNotOnLatestVersion = activeDocContext.activeVersion !== latestVersion;
    const latestDocSuggestion = isNotOnLatestVersion
        ? activeDocContext === null || activeDocContext === void 0 ? void 0 : activeDocContext.alternateDocVersions[latestVersion.name] : undefined;
    const latestVersionSuggestion = isNotOnLatestVersion
        ? latestVersion
        : undefined;
    return { latestDocSuggestion, latestVersionSuggestion };
};
