"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.processDocMetadata = exports.readVersionDocs = exports.readDocFile = void 0;
const path_1 = __importDefault(require("path"));
const fs_extra_1 = __importDefault(require("fs-extra"));
const utils_1 = require("@docusaurus/utils");
const lastUpdate_1 = require("./lastUpdate");
const slug_1 = __importDefault(require("./slug"));
const constants_1 = require("./constants");
const globby_1 = __importDefault(require("globby"));
async function readLastUpdateData(filePath, options) {
    const { showLastUpdateAuthor, showLastUpdateTime } = options;
    if (showLastUpdateAuthor || showLastUpdateTime) {
        // Use fake data in dev for faster development.
        const fileLastUpdateData = process.env.NODE_ENV === 'production'
            ? await lastUpdate_1.getFileLastUpdate(filePath)
            : {
                author: 'Author',
                timestamp: 1539502055,
            };
        if (fileLastUpdateData) {
            const { author, timestamp } = fileLastUpdateData;
            return {
                lastUpdatedAt: showLastUpdateTime ? timestamp : undefined,
                lastUpdatedBy: showLastUpdateAuthor ? author : undefined,
            };
        }
    }
    return {};
}
async function readDocFile(docsDirPath, source, options) {
    const filePath = path_1.default.join(docsDirPath, source);
    const [content, lastUpdate] = await Promise.all([
        fs_extra_1.default.readFile(filePath, 'utf-8'),
        readLastUpdateData(filePath, options),
    ]);
    return { source, content, lastUpdate };
}
exports.readDocFile = readDocFile;
async function readVersionDocs(versionMetadata, options) {
    const sources = await globby_1.default(options.include, {
        cwd: versionMetadata.docsDirPath,
    });
    return Promise.all(sources.map((source) => readDocFile(versionMetadata.docsDirPath, source, options)));
}
exports.readVersionDocs = readVersionDocs;
function processDocMetadata({ docFile, versionMetadata, context, options, }) {
    const { source, content, lastUpdate } = docFile;
    const { editUrl, homePageId } = options;
    const { siteDir } = context;
    const filePath = path_1.default.join(versionMetadata.docsDirPath, source);
    // ex: api/myDoc -> api
    // ex: myDoc -> .
    const docsFileDirName = path_1.default.dirname(source);
    const docsEditUrl = utils_1.getEditUrl(path_1.default.relative(siteDir, filePath), editUrl);
    const { frontMatter = {}, excerpt } = utils_1.parseMarkdownString(content);
    const { sidebar_label, custom_edit_url } = frontMatter;
    const baseID = frontMatter.id || path_1.default.basename(source, path_1.default.extname(source));
    if (baseID.includes('/')) {
        throw new Error(`Document id [${baseID}] cannot include "/".`);
    }
    // TODO legacy retrocompatibility
    // The same doc in 2 distinct version could keep the same id,
    // we just need to namespace the data by version
    const versionIdPart = versionMetadata.versionName === constants_1.CURRENT_VERSION_NAME
        ? ''
        : `version-${versionMetadata.versionName}/`;
    // TODO legacy retrocompatibility
    // I think it's bad to affect the frontmatter id with the dirname
    const dirNameIdPart = docsFileDirName === '.' ? '' : `${docsFileDirName}/`;
    // TODO legacy composite id, requires a breaking change to modify this
    const id = `${versionIdPart}${dirNameIdPart}${baseID}`;
    const unversionedId = `${dirNameIdPart}${baseID}`;
    // TODO remove soon, deprecated homePageId
    const isDocsHomePage = unversionedId === (homePageId !== null && homePageId !== void 0 ? homePageId : '_index');
    if (frontMatter.slug && isDocsHomePage) {
        throw new Error(`The docs homepage (homePageId=${homePageId}) is not allowed to have a frontmatter slug=${frontMatter.slug} => you have to chooser either homePageId or slug, not both`);
    }
    const docSlug = isDocsHomePage
        ? '/'
        : slug_1.default({
            baseID,
            dirName: docsFileDirName,
            frontmatterSlug: frontMatter.slug,
        });
    // Default title is the id.
    const title = frontMatter.title || baseID;
    const description = frontMatter.description || excerpt;
    const permalink = utils_1.normalizeUrl([versionMetadata.versionPath, docSlug]);
    // Assign all of object properties during instantiation (if possible) for
    // NodeJS optimization.
    // Adding properties to object after instantiation will cause hidden
    // class transitions.
    const metadata = {
        unversionedId,
        id,
        isDocsHomePage,
        title,
        description,
        source: utils_1.aliasedSitePath(filePath, siteDir),
        slug: docSlug,
        permalink,
        editUrl: custom_edit_url !== undefined ? custom_edit_url : docsEditUrl,
        version: versionMetadata.versionName,
        lastUpdatedBy: lastUpdate.lastUpdatedBy,
        lastUpdatedAt: lastUpdate.lastUpdatedAt,
        sidebar_label,
    };
    return metadata;
}
exports.processDocMetadata = processDocMetadata;
