"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.linkify = void 0;
const path_1 = __importDefault(require("path"));
const url_1 = require("url");
function getVersion(filePath, options) {
    const versionFound = options.versionsMetadata.find((version) => filePath.startsWith(version.docsDirPath));
    if (!versionFound) {
        throw new Error(`Unexpected, markdown file does not belong to any docs version! file=${filePath}`);
    }
    return versionFound;
}
function replaceMarkdownLinks(fileString, filePath, version, options) {
    const { siteDir, sourceToPermalink, onBrokenMarkdownLink } = options;
    const { docsDirPath } = version;
    // Replace internal markdown linking (except in fenced blocks).
    let fencedBlock = false;
    const lines = fileString.split('\n').map((line) => {
        if (line.trim().startsWith('```')) {
            fencedBlock = !fencedBlock;
        }
        if (fencedBlock) {
            return line;
        }
        let modifiedLine = line;
        // Replace inline-style links or reference-style links e.g:
        // This is [Document 1](doc1.md) -> we replace this doc1.md with correct link
        // [doc1]: doc1.md -> we replace this doc1.md with correct link
        const mdRegex = /(?:(?:\]\()|(?:\]:\s?))(?!https)([^'")\]\s>]+\.mdx?)/g;
        let mdMatch = mdRegex.exec(modifiedLine);
        while (mdMatch !== null) {
            // Replace it to correct html link.
            const mdLink = mdMatch[1];
            const targetSource = `${docsDirPath}/${mdLink}`;
            const aliasedSource = (source) => `@site/${path_1.default.relative(siteDir, source)}`;
            const permalink = sourceToPermalink[aliasedSource(url_1.resolve(filePath, mdLink))] ||
                sourceToPermalink[aliasedSource(targetSource)];
            if (permalink) {
                modifiedLine = modifiedLine.replace(mdLink, permalink);
            }
            else {
                const brokenMarkdownLink = {
                    version,
                    filePath,
                    link: mdLink,
                };
                onBrokenMarkdownLink(brokenMarkdownLink);
            }
            mdMatch = mdRegex.exec(modifiedLine);
        }
        return modifiedLine;
    });
    return lines.join('\n');
}
function linkify(fileString, filePath, options) {
    const version = getVersion(filePath, options);
    return replaceMarkdownLinks(fileString, filePath, version, options);
}
exports.linkify = linkify;
