"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSidebarsUtils = exports.collectSidebarsDocIds = exports.collectSidebarDocItems = exports.loadSidebars = void 0;
const lodash_flatmap_1 = __importDefault(require("lodash.flatmap"));
const fs_extra_1 = __importDefault(require("fs-extra"));
const import_fresh_1 = __importDefault(require("import-fresh"));
const lodash_1 = require("lodash");
const utils_1 = require("@docusaurus/utils");
function isCategoryShorthand(item) {
    return typeof item !== 'string' && !item.type;
}
// categories are collapsed by default, unless user set collapsed = false
const defaultCategoryCollapsedValue = true;
/**
 * Convert {category1: [item1,item2]} shorthand syntax to long-form syntax
 */
function normalizeCategoryShorthand(sidebar) {
    return Object.entries(sidebar).map(([label, items]) => ({
        type: 'category',
        collapsed: defaultCategoryCollapsedValue,
        label,
        items,
    }));
}
/**
 * Check that item contains only allowed keys.
 */
function assertItem(item, keys) {
    const unknownKeys = Object.keys(item).filter(
    // @ts-expect-error: key is always string
    (key) => !keys.includes(key) && key !== 'type');
    if (unknownKeys.length) {
        throw new Error(`Unknown sidebar item keys: ${unknownKeys}. Item: ${JSON.stringify(item)}`);
    }
}
function assertIsCategory(item) {
    assertItem(item, ['items', 'label', 'collapsed']);
    if (typeof item.label !== 'string') {
        throw new Error(`Error loading ${JSON.stringify(item)}. "label" must be a string.`);
    }
    if (!Array.isArray(item.items)) {
        throw new Error(`Error loading ${JSON.stringify(item)}. "items" must be an array.`);
    }
    // "collapsed" is an optional property
    if (item.hasOwnProperty('collapsed') && typeof item.collapsed !== 'boolean') {
        throw new Error(`Error loading ${JSON.stringify(item)}. "collapsed" must be a boolean.`);
    }
}
function assertIsDoc(item) {
    assertItem(item, ['id']);
    if (typeof item.id !== 'string') {
        throw new Error(`Error loading ${JSON.stringify(item)}. "id" must be a string.`);
    }
}
function assertIsLink(item) {
    assertItem(item, ['href', 'label']);
    if (typeof item.href !== 'string') {
        throw new Error(`Error loading ${JSON.stringify(item)}. "href" must be a string.`);
    }
    if (typeof item.label !== 'string') {
        throw new Error(`Error loading ${JSON.stringify(item)}. "label" must be a string.`);
    }
}
/**
 * Normalizes recursively item and all its children. Ensures that at the end
 * each item will be an object with the corresponding type.
 */
function normalizeItem(item) {
    if (typeof item === 'string') {
        return [
            {
                type: 'doc',
                id: item,
            },
        ];
    }
    if (isCategoryShorthand(item)) {
        return lodash_flatmap_1.default(normalizeCategoryShorthand(item), normalizeItem);
    }
    switch (item.type) {
        case 'category':
            assertIsCategory(item);
            return [
                Object.assign(Object.assign({ collapsed: defaultCategoryCollapsedValue }, item), { items: lodash_flatmap_1.default(item.items, normalizeItem) }),
            ];
        case 'link':
            assertIsLink(item);
            return [item];
        case 'ref':
        case 'doc':
            assertIsDoc(item);
            return [item];
        default: {
            const extraMigrationError = item.type === 'subcategory'
                ? "Docusaurus v2: 'subcategory' has been renamed as 'category'"
                : '';
            throw new Error(`Unknown sidebar item type [${item.type}]. Sidebar item=${JSON.stringify(item)} ${extraMigrationError}`);
        }
    }
}
function normalizeSidebar(sidebar) {
    const normalizedSidebar = Array.isArray(sidebar)
        ? sidebar
        : normalizeCategoryShorthand(sidebar);
    return lodash_flatmap_1.default(normalizedSidebar, normalizeItem);
}
function normalizeSidebars(sidebars) {
    return lodash_1.mapValues(sidebars, normalizeSidebar);
}
// TODO refactor: make async
function loadSidebars(sidebarFilePath) {
    if (!sidebarFilePath) {
        throw new Error(`sidebarFilePath not provided: ${sidebarFilePath}`);
    }
    // sidebars file is optional, some users use docs without sidebars!
    // See https://github.com/facebook/docusaurus/issues/3366
    if (!fs_extra_1.default.existsSync(sidebarFilePath)) {
        // throw new Error(`No sidebar file exist at path: ${sidebarFilePath}`);
        return {};
    }
    // We don't want sidebars to be cached because of hot reloading.
    const sidebarJson = import_fresh_1.default(sidebarFilePath);
    return normalizeSidebars(sidebarJson);
}
exports.loadSidebars = loadSidebars;
// traverse the sidebar tree in depth to find all doc items, in correct order
function collectSidebarDocItems(sidebar) {
    function collectRecursive(item) {
        if (item.type === 'doc') {
            return [item];
        }
        if (item.type === 'category') {
            return lodash_1.flatten(item.items.map(collectRecursive));
        }
        // Refs and links should not be shown in navigation.
        if (item.type === 'ref' || item.type === 'link') {
            return [];
        }
        throw new Error(`unknown sidebar item type = ${item.type}`);
    }
    return lodash_1.flatten(sidebar.map(collectRecursive));
}
exports.collectSidebarDocItems = collectSidebarDocItems;
function collectSidebarsDocIds(sidebars) {
    return lodash_1.mapValues(sidebars, (sidebar) => {
        return collectSidebarDocItems(sidebar).map((docItem) => docItem.id);
    });
}
exports.collectSidebarsDocIds = collectSidebarsDocIds;
function createSidebarsUtils(sidebars) {
    const sidebarNameToDocIds = collectSidebarsDocIds(sidebars);
    function getFirstDocIdOfFirstSidebar() {
        var _a;
        return (_a = Object.values(sidebarNameToDocIds)[0]) === null || _a === void 0 ? void 0 : _a[0];
    }
    function getSidebarNameByDocId(docId) {
        // TODO lookup speed can be optimized
        const entry = Object.entries(sidebarNameToDocIds).find(([_sidebarName, docIds]) => docIds.includes(docId));
        return entry === null || entry === void 0 ? void 0 : entry[0];
    }
    function getDocNavigation(docId) {
        const sidebarName = getSidebarNameByDocId(docId);
        if (sidebarName) {
            const docIds = sidebarNameToDocIds[sidebarName];
            const currentIndex = docIds.indexOf(docId);
            const { previous, next } = utils_1.getElementsAround(docIds, currentIndex);
            return {
                sidebarName,
                previousId: previous,
                nextId: next,
            };
        }
        else {
            return {
                sidebarName: undefined,
                previousId: undefined,
                nextId: undefined,
            };
        }
    }
    function checkSidebarsDocIds(validDocIds) {
        const allSidebarDocIds = lodash_1.flatten(Object.values(sidebarNameToDocIds));
        const invalidSidebarDocIds = lodash_1.difference(allSidebarDocIds, validDocIds);
        if (invalidSidebarDocIds.length > 0) {
            throw new Error(`Bad sidebars file.
These sidebar document ids do not exist:
- ${invalidSidebarDocIds.sort().join('\n- ')}\`,

Available document ids=
- ${validDocIds.sort().join('\n- ')}`);
        }
    }
    return {
        getFirstDocIdOfFirstSidebar,
        getSidebarNameByDocId,
        getDocNavigation,
        checkSidebarsDocIds,
    };
}
exports.createSidebarsUtils = createSidebarsUtils;
