"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateOptions = void 0;
const globby_1 = __importDefault(require("globby"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const minimatch_1 = __importDefault(require("minimatch"));
const slash_1 = __importDefault(require("slash"));
const utils_1 = require("@docusaurus/utils");
const remark_admonitions_1 = __importDefault(require("remark-admonitions"));
const pluginOptionSchema_1 = require("./pluginOptionSchema");
const constants_1 = require("@docusaurus/core/lib/constants");
const isMarkdownSource = (source) => source.endsWith('.md') || source.endsWith('.mdx');
function pluginContentPages(context, options) {
    var _a;
    if (options.admonitions) {
        options.remarkPlugins = options.remarkPlugins.concat([
            [remark_admonitions_1.default, options.admonitions || {}],
        ]);
    }
    const { siteConfig, siteDir, generatedFilesDir } = context;
    const contentPath = path_1.default.resolve(siteDir, options.path);
    const pluginDataDirRoot = path_1.default.join(generatedFilesDir, 'docusaurus-plugin-content-pages');
    const dataDir = path_1.default.join(pluginDataDirRoot, (_a = options.id) !== null && _a !== void 0 ? _a : constants_1.DEFAULT_PLUGIN_ID);
    const excludeRegex = new RegExp(options.exclude
        .map((pattern) => minimatch_1.default.makeRe(pattern).source)
        .join('|'));
    return {
        name: 'docusaurus-plugin-content-pages',
        getPathsToWatch() {
            const { include = [] } = options;
            const globPattern = include.map((pattern) => `${contentPath}/${pattern}`);
            return [...globPattern];
        },
        getClientModules() {
            const modules = [];
            if (options.admonitions) {
                modules.push(require.resolve('remark-admonitions/styles/infima.css'));
            }
            return modules;
        },
        async loadContent() {
            const { include } = options;
            const pagesDir = contentPath;
            if (!fs_1.default.existsSync(pagesDir)) {
                return null;
            }
            const { baseUrl } = siteConfig;
            const pagesFiles = await globby_1.default(include, {
                cwd: pagesDir,
                ignore: options.exclude,
            });
            function toMetadata(relativeSource) {
                const source = path_1.default.join(pagesDir, relativeSource);
                const aliasedSourcePath = utils_1.aliasedSitePath(source, siteDir);
                const pathName = utils_1.encodePath(utils_1.fileToPath(relativeSource));
                const permalink = pathName.replace(/^\//, baseUrl || '');
                if (isMarkdownSource(relativeSource)) {
                    return {
                        type: 'mdx',
                        permalink,
                        source: aliasedSourcePath,
                    };
                }
                else {
                    return {
                        type: 'jsx',
                        permalink,
                        source: aliasedSourcePath,
                    };
                }
            }
            return pagesFiles.map(toMetadata);
        },
        async contentLoaded({ content, actions }) {
            if (!content) {
                return;
            }
            const { addRoute, createData } = actions;
            await Promise.all(content.map(async (metadata) => {
                const { permalink, source } = metadata;
                if (metadata.type === 'mdx') {
                    await createData(
                    // Note that this created data path must be in sync with
                    // metadataPath provided to mdx-loader.
                    `${utils_1.docuHash(metadata.source)}.json`, JSON.stringify(metadata, null, 2));
                    addRoute({
                        path: permalink,
                        component: options.mdxPageComponent,
                        exact: true,
                        modules: {
                            content: source,
                        },
                    });
                }
                else {
                    addRoute({
                        path: permalink,
                        component: source,
                        exact: true,
                        modules: {
                            config: `@generated/docusaurus.config`,
                        },
                    });
                }
            }));
        },
        configureWebpack(_config, isServer, { getBabelLoader, getCacheLoader }) {
            const { rehypePlugins, remarkPlugins } = options;
            return {
                resolve: {
                    alias: {
                        '~pages': pluginDataDirRoot,
                    },
                },
                module: {
                    rules: [
                        {
                            test: /(\.mdx?)$/,
                            include: [contentPath],
                            use: [
                                getCacheLoader(isServer),
                                getBabelLoader(isServer),
                                {
                                    loader: require.resolve('@docusaurus/mdx-loader'),
                                    options: {
                                        remarkPlugins,
                                        rehypePlugins,
                                        staticDir: path_1.default.join(siteDir, constants_1.STATIC_DIR_NAME),
                                        // Note that metadataPath must be the same/in-sync as
                                        // the path from createData for each MDX.
                                        metadataPath: (mdxPath) => {
                                            if (excludeRegex.test(slash_1.default(mdxPath))) {
                                                return null;
                                            }
                                            const aliasedSource = utils_1.aliasedSitePath(mdxPath, siteDir);
                                            return path_1.default.join(dataDir, `${utils_1.docuHash(aliasedSource)}.json`);
                                        },
                                        forbidFrontMatter: (mdxPath) => excludeRegex.test(slash_1.default(mdxPath)),
                                    },
                                },
                                {
                                    loader: path_1.default.resolve(__dirname, './markdownLoader.js'),
                                    options: {
                                    // siteDir,
                                    // contentPath,
                                    },
                                },
                            ].filter(Boolean),
                        },
                    ],
                },
            };
        },
    };
}
exports.default = pluginContentPages;
function validateOptions({ validate, options, }) {
    const validatedOptions = validate(pluginOptionSchema_1.PluginOptionSchema, options);
    return validatedOptions;
}
exports.validateOptions = validateOptions;
