"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getElementsAround = exports.getFilePathForRoutePath = exports.removePrefix = exports.removeSuffix = exports.removeTrailingSlash = exports.addTrailingSlash = exports.addLeadingSlash = exports.resolvePathname = exports.isValidPathname = exports.getEditUrl = exports.aliasedSitePath = exports.normalizeUrl = exports.parseMarkdownFile = exports.parseMarkdownString = exports.createExcerpt = exports.getSubFolder = exports.idx = exports.genChunkName = exports.posixPath = exports.genComponentName = exports.upperFirst = exports.docuHash = exports.simpleHash = exports.encodePath = exports.fileToPath = exports.objectWithKeySorted = exports.generate = void 0;
const path_1 = __importDefault(require("path"));
const gray_matter_1 = __importDefault(require("gray-matter"));
const crypto_1 = require("crypto");
const lodash_camelcase_1 = __importDefault(require("lodash.camelcase"));
const lodash_kebabcase_1 = __importDefault(require("lodash.kebabcase"));
const escape_string_regexp_1 = __importDefault(require("escape-string-regexp"));
const fs_extra_1 = __importDefault(require("fs-extra"));
const url_1 = require("url");
// @ts-expect-error: no typedefs :s
const resolve_pathname_1 = __importDefault(require("resolve-pathname"));
const fileHash = new Map();
async function generate(generatedFilesDir, file, content, skipCache = process.env.NODE_ENV === 'production') {
    const filepath = path_1.default.join(generatedFilesDir, file);
    if (skipCache) {
        await fs_extra_1.default.ensureDir(path_1.default.dirname(filepath));
        await fs_extra_1.default.writeFile(filepath, content);
        return;
    }
    let lastHash = fileHash.get(filepath);
    // If file already exists but its not in runtime cache yet,
    // we try to calculate the content hash and then compare
    // This is to avoid unnecessary overwriting and we can reuse old file.
    if (!lastHash && fs_extra_1.default.existsSync(filepath)) {
        const lastContent = await fs_extra_1.default.readFile(filepath, 'utf8');
        lastHash = crypto_1.createHash('md5').update(lastContent).digest('hex');
        fileHash.set(filepath, lastHash);
    }
    const currentHash = crypto_1.createHash('md5').update(content).digest('hex');
    if (lastHash !== currentHash) {
        await fs_extra_1.default.ensureDir(path_1.default.dirname(filepath));
        await fs_extra_1.default.writeFile(filepath, content);
        fileHash.set(filepath, currentHash);
    }
}
exports.generate = generate;
function objectWithKeySorted(obj) {
    // https://github.com/lodash/lodash/issues/1459#issuecomment-460941233
    return Object.keys(obj)
        .sort()
        .reduce((acc, key) => {
        acc[key] = obj[key];
        return acc;
    }, {});
}
exports.objectWithKeySorted = objectWithKeySorted;
const indexRE = /(^|.*\/)index\.(md|mdx|js|jsx|ts|tsx)$/i;
const extRE = /\.(md|mdx|js|jsx|ts|tsx)$/;
/**
 * Convert filepath to url path.
 * Example: 'index.md' -> '/', 'foo/bar.js' -> '/foo/bar',
 */
function fileToPath(file) {
    if (indexRE.test(file)) {
        return file.replace(indexRE, '/$1');
    }
    return `/${file.replace(extRE, '').replace(/\\/g, '/')}`;
}
exports.fileToPath = fileToPath;
function encodePath(userpath) {
    return userpath
        .split('/')
        .map((item) => encodeURIComponent(item))
        .join('/');
}
exports.encodePath = encodePath;
function simpleHash(str, length) {
    return crypto_1.createHash('md5').update(str).digest('hex').substr(0, length);
}
exports.simpleHash = simpleHash;
/**
 * Given an input string, convert to kebab-case and append a hash.
 * Avoid str collision.
 */
function docuHash(str) {
    if (str === '/') {
        return 'index';
    }
    const shortHash = simpleHash(str, 3);
    return `${lodash_kebabcase_1.default(str)}-${shortHash}`;
}
exports.docuHash = docuHash;
/**
 * Convert first string character to the upper case.
 * E.g: docusaurus -> Docusaurus
 */
function upperFirst(str) {
    return str ? str.charAt(0).toUpperCase() + str.slice(1) : '';
}
exports.upperFirst = upperFirst;
/**
 * Generate unique React Component Name.
 * E.g: /foo-bar -> FooBar096
 */
function genComponentName(pagePath) {
    if (pagePath === '/') {
        return 'index';
    }
    const pageHash = docuHash(pagePath);
    return upperFirst(lodash_camelcase_1.default(pageHash));
}
exports.genComponentName = genComponentName;
/**
 * Convert Windows backslash paths to posix style paths.
 * E.g: endi\\lie -> endi/lie
 */
function posixPath(str) {
    const isExtendedLengthPath = /^\\\\\?\\/.test(str);
    const hasNonAscii = /[^\u0000-\u0080]+/.test(str); // eslint-disable-line
    if (isExtendedLengthPath || hasNonAscii) {
        return str;
    }
    return str.replace(/\\/g, '/');
}
exports.posixPath = posixPath;
const chunkNameCache = new Map();
/**
 * Generate unique chunk name given a module path.
 */
function genChunkName(modulePath, prefix, preferredName, shortId = process.env.NODE_ENV === 'production') {
    let chunkName = chunkNameCache.get(modulePath);
    if (!chunkName) {
        if (shortId) {
            chunkName = simpleHash(modulePath, 8);
        }
        else {
            let str = modulePath;
            if (preferredName) {
                const shortHash = simpleHash(modulePath, 3);
                str = `${preferredName}${shortHash}`;
            }
            const name = str === '/' ? 'index' : docuHash(str);
            chunkName = prefix ? `${prefix}---${name}` : name;
        }
        chunkNameCache.set(modulePath, chunkName);
    }
    return chunkName;
}
exports.genChunkName = genChunkName;
// Too dynamic
// eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types, @typescript-eslint/no-explicit-any
function idx(target, keyPaths) {
    return (target &&
        keyPaths &&
        (Array.isArray(keyPaths)
            ? keyPaths.reduce((obj, key) => obj && obj[key], target)
            : target[keyPaths]));
}
exports.idx = idx;
/**
 * Given a filepath and dirpath, get the first directory.
 */
function getSubFolder(file, refDir) {
    const separator = escape_string_regexp_1.default(path_1.default.sep);
    const baseDir = escape_string_regexp_1.default(path_1.default.basename(refDir));
    const regexSubFolder = new RegExp(`${baseDir}${separator}(.*?)${separator}.*`);
    const match = regexSubFolder.exec(file);
    return match && match[1];
}
exports.getSubFolder = getSubFolder;
// Regex for an import statement.
const importRegexString = '^(.*import){1}(.+){0,1}\\s[\'"](.+)[\'"];?';
function createExcerpt(fileString) {
    let fileContent = fileString.trimLeft();
    if (RegExp(importRegexString).test(fileContent)) {
        fileContent = fileContent
            .replace(RegExp(importRegexString, 'gm'), '')
            .trimLeft();
    }
    const fileLines = fileContent.split('\n');
    for (const fileLine of fileLines) {
        const cleanedLine = fileLine
            // Remove HTML tags.
            .replace(/<[^>]*>/g, '')
            // Remove ATX-style headers.
            .replace(/^\#{1,6}\s*([^#]*)\s*(\#{1,6})?/gm, '$1')
            // Remove emphasis and strikethroughs.
            .replace(/([\*_~]{1,3})(\S.*?\S{0,1})\1/g, '$2')
            // Remove images.
            .replace(/\!\[(.*?)\][\[\(].*?[\]\)]/g, '$1')
            // Remove footnotes.
            .replace(/\[\^.+?\](\: .*?$)?/g, '')
            // Remove inline links.
            .replace(/\[(.*?)\][\[\(].*?[\]\)]/g, '$1')
            // Remove inline code.
            .replace(/`(.+?)`/g, '$1')
            // Remove blockquotes.
            .replace(/^\s{0,3}>\s?/g, '')
            // Remove admonition definition.
            .replace(/(:{3}.*)/, '')
            // Remove Emoji names within colons include preceding whitespace.
            .replace(/\s?(:(::|[^:\n])+:)/g, '')
            .trim();
        if (cleanedLine) {
            return cleanedLine;
        }
    }
    return undefined;
}
exports.createExcerpt = createExcerpt;
function parseMarkdownString(markdownString) {
    const options = {
        excerpt: (file) => {
            // Hacky way of stripping out import statements from the excerpt
            // TODO: Find a better way to do so, possibly by compiling the Markdown content,
            // stripping out HTML tags and obtaining the first line.
            file.excerpt = createExcerpt(file.content);
        },
    };
    try {
        const { data: frontMatter, content, excerpt } = gray_matter_1.default(markdownString, options);
        return { frontMatter, content, excerpt };
    }
    catch (e) {
        throw new Error(`Error while parsing markdown front matter.
This can happen if you use special characteres like : in frontmatter values (try using "" around that value)
${e.message}`);
    }
}
exports.parseMarkdownString = parseMarkdownString;
async function parseMarkdownFile(source) {
    const markdownString = await fs_extra_1.default.readFile(source, 'utf-8');
    try {
        return parseMarkdownString(markdownString);
    }
    catch (e) {
        throw new Error(`Error while parsing markdown file ${source}
${e.message}`);
    }
}
exports.parseMarkdownFile = parseMarkdownFile;
function normalizeUrl(rawUrls) {
    const urls = rawUrls;
    const resultArray = [];
    // If the first part is a plain protocol, we combine it with the next part.
    if (urls[0].match(/^[^/:]+:\/*$/) && urls.length > 1) {
        const first = urls.shift();
        urls[0] = first + urls[0];
    }
    // There must be two or three slashes in the file protocol,
    // two slashes in anything else.
    const replacement = urls[0].match(/^file:\/\/\//) ? '$1:///' : '$1://';
    urls[0] = urls[0].replace(/^([^/:]+):\/*/, replacement);
    // eslint-disable-next-line
    for (let i = 0; i < urls.length; i++) {
        let component = urls[i];
        if (typeof component !== 'string') {
            throw new TypeError(`Url must be a string. Received ${typeof component}`);
        }
        if (component === '') {
            // eslint-disable-next-line
            continue;
        }
        if (i > 0) {
            // Removing the starting slashes for each component but the first.
            component = component.replace(/^[/]+/, '');
        }
        // Removing the ending slashes for each component but the last.
        // For the last component we will combine multiple slashes to a single one.
        component = component.replace(/[/]+$/, i < urls.length - 1 ? '' : '/');
        resultArray.push(component);
    }
    let str = resultArray.join('/');
    // Each input component is now separated by a single slash
    // except the possible first plain protocol part.
    // Remove trailing slash before parameters or hash.
    str = str.replace(/\/(\?|&|#[^!])/g, '$1');
    // Replace ? in parameters with &.
    const parts = str.split('?');
    str = parts.shift() + (parts.length > 0 ? '?' : '') + parts.join('&');
    // Dedupe forward slashes in the entire path, avoiding protocol slashes.
    str = str.replace(/([^:]\/)\/+/g, '$1');
    // Dedupe forward slashes at the beginning of the path.
    str = str.replace(/^\/+/g, '/');
    return str;
}
exports.normalizeUrl = normalizeUrl;
/**
 * Alias filepath relative to site directory, very useful so that we
 * don't expose user's site structure.
 * Example: some/path/to/website/docs/foo.md -> @site/docs/foo.md
 */
function aliasedSitePath(filePath, siteDir) {
    const relativePath = path_1.default.relative(siteDir, filePath);
    // Cannot use path.join() as it resolves '../' and removes
    // the '@site'. Let webpack loader resolve it.
    return `@site/${relativePath}`;
}
exports.aliasedSitePath = aliasedSitePath;
function getEditUrl(fileRelativePath, editUrl) {
    return editUrl
        ? normalizeUrl([editUrl, posixPath(fileRelativePath)])
        : undefined;
}
exports.getEditUrl = getEditUrl;
function isValidPathname(str) {
    if (!str.startsWith('/')) {
        return false;
    }
    try {
        // weird, but is there a better way?
        const parsedPathname = new url_1.URL(str, 'https://domain.com').pathname;
        return parsedPathname === str || parsedPathname === encodeURI(str);
    }
    catch (e) {
        return false;
    }
}
exports.isValidPathname = isValidPathname;
// resolve pathname and fail fast if resolution fails
function resolvePathname(to, from) {
    return resolve_pathname_1.default(to, from);
}
exports.resolvePathname = resolvePathname;
function addLeadingSlash(str) {
    return str.startsWith('/') ? str : `/${str}`;
}
exports.addLeadingSlash = addLeadingSlash;
function addTrailingSlash(str) {
    return str.endsWith('/') ? str : `${str}/`;
}
exports.addTrailingSlash = addTrailingSlash;
function removeTrailingSlash(str) {
    return removeSuffix(str, '/');
}
exports.removeTrailingSlash = removeTrailingSlash;
function removeSuffix(str, suffix) {
    if (suffix === '') {
        return str; // always returns "" otherwise!
    }
    return str.endsWith(suffix) ? str.slice(0, -suffix.length) : str;
}
exports.removeSuffix = removeSuffix;
function removePrefix(str, prefix) {
    return str.startsWith(prefix) ? str.slice(prefix.length) : str;
}
exports.removePrefix = removePrefix;
function getFilePathForRoutePath(routePath) {
    const fileName = path_1.default.basename(routePath);
    const filePath = path_1.default.dirname(routePath);
    return path_1.default.join(filePath, `${fileName}/index.html`);
}
exports.getFilePathForRoutePath = getFilePathForRoutePath;
function getElementsAround(array, aroundIndex) {
    const min = 0;
    const max = array.length - 1;
    if (aroundIndex < min || aroundIndex > max) {
        throw new Error(`Valid aroundIndex for array (of size ${array.length}) are between ${min} and ${max}, but you provided aroundIndex=${aroundIndex}`);
    }
    const previous = aroundIndex === min ? undefined : array[aroundIndex - 1];
    const next = aroundIndex === max ? undefined : array[aroundIndex + 1];
    return { previous, next };
}
exports.getElementsAround = getElementsAround;
