"use strict";
/* eslint-disable no-dupe-class-members */
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var buffer_1 = require("buffer");
var ethereumjs_util_1 = require("ethereumjs-util");
var common_1 = __importDefault(require("@ethereumjs/common"));
// secp256k1n/2
var N_DIV_2 = new ethereumjs_util_1.BN('7fffffffffffffffffffffffffffffff5d576e7357a4501ddfe92f46681b20a0', 16);
/**
 * An Ethereum transaction.
 */
var Transaction = /** @class */ (function () {
    /**
     * This constructor takes the values, validates them, assigns them and freezes the object.
     * Use the static factory methods to assist in creating a Transaction object from varying data types.
     * @note Transaction objects implement EIP155 by default. To disable it, pass in an `@ethereumjs/common` object set before EIP155 activation (i.e. before Spurious Dragon).
     */
    function Transaction(nonce, gasPrice, gasLimit, to, value, data, v, r, s, opts) {
        var e_1, _a;
        var validateCannotExceedMaxInteger = {
            nonce: nonce,
            gasPrice: gasPrice,
            gasLimit: gasLimit,
            value: value,
            r: r,
            s: s,
        };
        try {
            for (var _b = __values(Object.entries(validateCannotExceedMaxInteger)), _c = _b.next(); !_c.done; _c = _b.next()) {
                var _d = __read(_c.value, 2), key = _d[0], value_1 = _d[1];
                if (value_1 && value_1.gt(ethereumjs_util_1.MAX_INTEGER)) {
                    throw new Error(key + " cannot exceed MAX_INTEGER, given " + value_1);
                }
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_1) throw e_1.error; }
        }
        if (opts === null || opts === void 0 ? void 0 : opts.common) {
            this.common = opts.common;
        }
        else {
            var DEFAULT_CHAIN = 'mainnet';
            this.common = new common_1.default({ chain: DEFAULT_CHAIN });
        }
        this._validateTxV(v);
        this.nonce = nonce;
        this.gasPrice = gasPrice;
        this.gasLimit = gasLimit;
        this.to = to;
        this.value = value;
        this.data = data;
        this.v = v;
        this.r = r;
        this.s = s;
        Object.freeze(this);
    }
    Transaction.fromTxData = function (txData, opts) {
        var nonce = txData.nonce, gasLimit = txData.gasLimit, gasPrice = txData.gasPrice, to = txData.to, value = txData.value, data = txData.data, v = txData.v, r = txData.r, s = txData.s;
        return new Transaction(new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(nonce)), new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(gasPrice)), new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(gasLimit)), to ? new ethereumjs_util_1.Address(ethereumjs_util_1.toBuffer(to)) : undefined, new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(value)), ethereumjs_util_1.toBuffer(data), new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(v)), new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(r)), new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(s)), opts);
    };
    Transaction.fromRlpSerializedTx = function (serialized, opts) {
        var values = ethereumjs_util_1.rlp.decode(serialized);
        if (!Array.isArray(values)) {
            throw new Error('Invalid serialized tx input. Must be array');
        }
        return this.fromValuesArray(values, opts);
    };
    Transaction.fromValuesArray = function (values, opts) {
        if (values.length !== 6 && values.length !== 9) {
            throw new Error('Invalid transaction. Only expecting 6 values (for unsigned tx) or 9 values (for signed tx).');
        }
        var _a = __read(values, 9), nonce = _a[0], gasPrice = _a[1], gasLimit = _a[2], to = _a[3], value = _a[4], data = _a[5], v = _a[6], r = _a[7], s = _a[8];
        return new Transaction(new ethereumjs_util_1.BN(nonce), new ethereumjs_util_1.BN(gasPrice), new ethereumjs_util_1.BN(gasLimit), to && to.length > 0 ? new ethereumjs_util_1.Address(to) : undefined, new ethereumjs_util_1.BN(value), data || buffer_1.Buffer.from([]), v ? new ethereumjs_util_1.BN(v) : undefined, r ? new ethereumjs_util_1.BN(r) : undefined, s ? new ethereumjs_util_1.BN(s) : undefined, opts);
    };
    /**
     * If the tx's `to` is to the creation address
     */
    Transaction.prototype.toCreationAddress = function () {
        return this.to === undefined || this.to.buf.length === 0;
    };
    /**
     * Computes a sha3-256 hash of the serialized tx
     */
    Transaction.prototype.hash = function () {
        var values = [
            ethereumjs_util_1.bnToRlp(this.nonce),
            ethereumjs_util_1.bnToRlp(this.gasPrice),
            ethereumjs_util_1.bnToRlp(this.gasLimit),
            this.to !== undefined ? this.to.buf : buffer_1.Buffer.from([]),
            ethereumjs_util_1.bnToRlp(this.value),
            this.data,
            this.v ? ethereumjs_util_1.bnToRlp(this.v) : buffer_1.Buffer.from([]),
            this.r ? ethereumjs_util_1.bnToRlp(this.r) : buffer_1.Buffer.from([]),
            this.s ? ethereumjs_util_1.bnToRlp(this.s) : buffer_1.Buffer.from([]),
        ];
        return ethereumjs_util_1.rlphash(values);
    };
    Transaction.prototype.getMessageToSign = function () {
        return this._getMessageToSign(this._unsignedTxImplementsEIP155());
    };
    Transaction.prototype.getMessageToVerifySignature = function () {
        return this._getMessageToSign(this._signedTxImplementsEIP155());
    };
    /**
     * Returns chain ID
     */
    Transaction.prototype.getChainId = function () {
        return this.common.chainId();
    };
    /**
     * Returns the sender's address
     */
    Transaction.prototype.getSenderAddress = function () {
        return new ethereumjs_util_1.Address(ethereumjs_util_1.publicToAddress(this.getSenderPublicKey()));
    };
    /**
     * Returns the public key of the sender
     */
    Transaction.prototype.getSenderPublicKey = function () {
        var msgHash = this.getMessageToVerifySignature();
        // All transaction signatures whose s-value is greater than secp256k1n/2 are considered invalid.
        if (this.common.gteHardfork('homestead') && this.s && this.s.gt(N_DIV_2)) {
            throw new Error('Invalid Signature: s-values greater than secp256k1n/2 are considered invalid');
        }
        var _a = this, v = _a.v, r = _a.r, s = _a.s;
        if (!v || !r || !s) {
            throw new Error('Missing values to derive sender public key from signed tx');
        }
        try {
            return ethereumjs_util_1.ecrecover(msgHash, v.toNumber(), ethereumjs_util_1.bnToRlp(r), ethereumjs_util_1.bnToRlp(s), this._signedTxImplementsEIP155() ? this.getChainId() : undefined);
        }
        catch (e) {
            throw new Error('Invalid Signature');
        }
    };
    /**
     * Determines if the signature is valid
     */
    Transaction.prototype.verifySignature = function () {
        try {
            return ethereumjs_util_1.unpadBuffer(this.getSenderPublicKey()).length !== 0;
        }
        catch (e) {
            return false;
        }
    };
    /**
     * Sign a transaction with a given private key.
     * Returns a new Transaction object (the original tx will not be modified).
     * Example:
     * ```typescript
     * const unsignedTx = Transaction.fromTxData(txData)
     * const signedTx = unsignedTx.sign(privKey)
     * ```
     * @param privateKey Must be 32 bytes in length.
     */
    Transaction.prototype.sign = function (privateKey) {
        if (privateKey.length !== 32) {
            throw new Error('Private key must be 32 bytes in length.');
        }
        var msgHash = this.getMessageToSign();
        // Only `v` is reassigned.
        /* eslint-disable-next-line prefer-const */
        var _a = ethereumjs_util_1.ecsign(msgHash, privateKey), v = _a.v, r = _a.r, s = _a.s;
        if (this._unsignedTxImplementsEIP155()) {
            v += this.getChainId() * 2 + 8;
        }
        var opts = {
            common: this.common,
        };
        return new Transaction(this.nonce, this.gasPrice, this.gasLimit, this.to, this.value, this.data, new ethereumjs_util_1.BN(v), new ethereumjs_util_1.BN(r), new ethereumjs_util_1.BN(s), opts);
    };
    /**
     * The amount of gas paid for the data in this tx
     */
    Transaction.prototype.getDataFee = function () {
        var txDataZero = this.common.param('gasPrices', 'txDataZero');
        var txDataNonZero = this.common.param('gasPrices', 'txDataNonZero');
        var cost = 0;
        for (var i = 0; i < this.data.length; i++) {
            this.data[i] === 0 ? (cost += txDataZero) : (cost += txDataNonZero);
        }
        return new ethereumjs_util_1.BN(cost);
    };
    /**
     * The minimum amount of gas the tx must have (DataFee + TxFee + Creation Fee)
     */
    Transaction.prototype.getBaseFee = function () {
        var fee = this.getDataFee().addn(this.common.param('gasPrices', 'tx'));
        if (this.common.gteHardfork('homestead') && this.toCreationAddress()) {
            fee.iaddn(this.common.param('gasPrices', 'txCreation'));
        }
        return fee;
    };
    /**
     * The up front amount that an account must have for this transaction to be valid
     */
    Transaction.prototype.getUpfrontCost = function () {
        return this.gasLimit.mul(this.gasPrice).add(this.value);
    };
    Transaction.prototype.validate = function (stringError) {
        if (stringError === void 0) { stringError = false; }
        var errors = [];
        if (!this.verifySignature()) {
            errors.push('Invalid Signature');
        }
        if (this.getBaseFee().gt(this.gasLimit)) {
            errors.push("gasLimit is too low. given " + this.gasLimit + ", need at least " + this.getBaseFee());
        }
        return stringError ? errors : errors.length === 0;
    };
    /**
     * Returns a Buffer Array of the raw Buffers of this transaction, in order.
     */
    Transaction.prototype.raw = function () {
        return [
            ethereumjs_util_1.bnToRlp(this.nonce),
            ethereumjs_util_1.bnToRlp(this.gasPrice),
            ethereumjs_util_1.bnToRlp(this.gasLimit),
            this.to !== undefined ? this.to.buf : buffer_1.Buffer.from([]),
            ethereumjs_util_1.bnToRlp(this.value),
            this.data,
            this.v !== undefined ? ethereumjs_util_1.bnToRlp(this.v) : buffer_1.Buffer.from([]),
            this.r !== undefined ? ethereumjs_util_1.bnToRlp(this.r) : buffer_1.Buffer.from([]),
            this.s !== undefined ? ethereumjs_util_1.bnToRlp(this.s) : buffer_1.Buffer.from([]),
        ];
    };
    /**
     * Returns the rlp encoding of the transaction.
     */
    Transaction.prototype.serialize = function () {
        return ethereumjs_util_1.rlp.encode(this.raw());
    };
    /**
     * Returns an object with the JSON representation of the transaction
     */
    Transaction.prototype.toJSON = function () {
        return {
            nonce: ethereumjs_util_1.bnToHex(this.nonce),
            gasPrice: ethereumjs_util_1.bnToHex(this.gasPrice),
            gasLimit: ethereumjs_util_1.bnToHex(this.gasLimit),
            to: this.to !== undefined ? this.to.toString() : undefined,
            value: ethereumjs_util_1.bnToHex(this.value),
            data: '0x' + this.data.toString('hex'),
            v: this.v !== undefined ? ethereumjs_util_1.bnToHex(this.v) : undefined,
            r: this.r !== undefined ? ethereumjs_util_1.bnToHex(this.r) : undefined,
            s: this.s !== undefined ? ethereumjs_util_1.bnToHex(this.s) : undefined,
        };
    };
    Transaction.prototype.isSigned = function () {
        var _a = this, v = _a.v, r = _a.r, s = _a.s;
        return !!v && !!r && !!s;
    };
    Transaction.prototype._unsignedTxImplementsEIP155 = function () {
        return this.common.gteHardfork('spuriousDragon');
    };
    Transaction.prototype._signedTxImplementsEIP155 = function () {
        var _a;
        if (!this.isSigned()) {
            throw Error('This transaction is not signed');
        }
        var onEIP155BlockOrLater = this.common.gteHardfork('spuriousDragon');
        // EIP155 spec:
        // If block.number >= 2,675,000 and v = CHAIN_ID * 2 + 35 or v = CHAIN_ID * 2 + 36, then when computing the hash of a transaction for purposes of signing or recovering, instead of hashing only the first six elements (i.e. nonce, gasprice, startgas, to, value, data), hash nine elements, with v replaced by CHAIN_ID, r = 0 and s = 0.
        var v = (_a = this.v) === null || _a === void 0 ? void 0 : _a.toNumber();
        var vAndChainIdMeetEIP155Conditions = v === this.getChainId() * 2 + 35 || v === this.getChainId() * 2 + 36;
        return vAndChainIdMeetEIP155Conditions && onEIP155BlockOrLater;
    };
    Transaction.prototype._getMessageToSign = function (withEIP155) {
        var values = [
            ethereumjs_util_1.bnToRlp(this.nonce),
            ethereumjs_util_1.bnToRlp(this.gasPrice),
            ethereumjs_util_1.bnToRlp(this.gasLimit),
            this.to !== undefined ? this.to.buf : buffer_1.Buffer.from([]),
            ethereumjs_util_1.bnToRlp(this.value),
            this.data,
        ];
        if (withEIP155) {
            values.push(ethereumjs_util_1.toBuffer(this.getChainId()));
            values.push(ethereumjs_util_1.unpadBuffer(ethereumjs_util_1.toBuffer(0)));
            values.push(ethereumjs_util_1.unpadBuffer(ethereumjs_util_1.toBuffer(0)));
        }
        return ethereumjs_util_1.rlphash(values);
    };
    /**
     * Validates tx's `v` value
     */
    Transaction.prototype._validateTxV = function (v) {
        if (v === undefined || v.toNumber() === 0) {
            return;
        }
        if (!this.common.gteHardfork('spuriousDragon')) {
            return;
        }
        var vInt = v.toNumber();
        if (vInt === 27 || vInt === 28) {
            return;
        }
        var isValidEIP155V = vInt === this.getChainId() * 2 + 35 || vInt === this.getChainId() * 2 + 36;
        if (!isValidEIP155V) {
            throw new Error("Incompatible EIP155-based V " + vInt + " and chain id " + this.getChainId() + ". See the Common parameter of the Transaction constructor to set the chain id.");
        }
    };
    return Transaction;
}());
exports.default = Transaction;
//# sourceMappingURL=transaction.js.map