/// <reference types="bn.js" />
/// <reference types="node" />
import { Address, BN } from 'ethereumjs-util';
import Common from '@ethereumjs/common';
import { TxOptions, TxData, JsonTx } from './types';
/**
 * An Ethereum transaction.
 */
export default class Transaction {
    readonly common: Common;
    readonly nonce: BN;
    readonly gasLimit: BN;
    readonly gasPrice: BN;
    readonly to?: Address;
    readonly value: BN;
    readonly data: Buffer;
    readonly v?: BN;
    readonly r?: BN;
    readonly s?: BN;
    static fromTxData(txData: TxData, opts?: TxOptions): Transaction;
    static fromRlpSerializedTx(serialized: Buffer, opts?: TxOptions): Transaction;
    static fromValuesArray(values: Buffer[], opts?: TxOptions): Transaction;
    /**
     * This constructor takes the values, validates them, assigns them and freezes the object.
     * Use the static factory methods to assist in creating a Transaction object from varying data types.
     * @note Transaction objects implement EIP155 by default. To disable it, pass in an `@ethereumjs/common` object set before EIP155 activation (i.e. before Spurious Dragon).
     */
    constructor(nonce: BN, gasPrice: BN, gasLimit: BN, to: Address | undefined, value: BN, data: Buffer, v?: BN, r?: BN, s?: BN, opts?: TxOptions);
    /**
     * If the tx's `to` is to the creation address
     */
    toCreationAddress(): boolean;
    /**
     * Computes a sha3-256 hash of the serialized tx
     */
    hash(): Buffer;
    getMessageToSign(): Buffer;
    getMessageToVerifySignature(): Buffer;
    /**
     * Returns chain ID
     */
    getChainId(): number;
    /**
     * Returns the sender's address
     */
    getSenderAddress(): Address;
    /**
     * Returns the public key of the sender
     */
    getSenderPublicKey(): Buffer;
    /**
     * Determines if the signature is valid
     */
    verifySignature(): boolean;
    /**
     * Sign a transaction with a given private key.
     * Returns a new Transaction object (the original tx will not be modified).
     * Example:
     * ```typescript
     * const unsignedTx = Transaction.fromTxData(txData)
     * const signedTx = unsignedTx.sign(privKey)
     * ```
     * @param privateKey Must be 32 bytes in length.
     */
    sign(privateKey: Buffer): Transaction;
    /**
     * The amount of gas paid for the data in this tx
     */
    getDataFee(): BN;
    /**
     * The minimum amount of gas the tx must have (DataFee + TxFee + Creation Fee)
     */
    getBaseFee(): BN;
    /**
     * The up front amount that an account must have for this transaction to be valid
     */
    getUpfrontCost(): BN;
    /**
     * Validates the signature and checks to see if it has enough gas.
     */
    validate(): boolean;
    validate(stringError: false): boolean;
    validate(stringError: true): string[];
    /**
     * Returns a Buffer Array of the raw Buffers of this transaction, in order.
     */
    raw(): Buffer[];
    /**
     * Returns the rlp encoding of the transaction.
     */
    serialize(): Buffer;
    /**
     * Returns an object with the JSON representation of the transaction
     */
    toJSON(): JsonTx;
    isSigned(): boolean;
    private _unsignedTxImplementsEIP155;
    private _signedTxImplementsEIP155;
    private _getMessageToSign;
    /**
     * Validates tx's `v` value
     */
    private _validateTxV;
}
