"use strict";
/* eslint-disable no-dupe-class-members */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const buffer_1 = require("buffer");
const ethereumjs_util_1 = require("ethereumjs-util");
const common_1 = __importDefault(require("@ethereumjs/common"));
// secp256k1n/2
const N_DIV_2 = new ethereumjs_util_1.BN('7fffffffffffffffffffffffffffffff5d576e7357a4501ddfe92f46681b20a0', 16);
/**
 * An Ethereum transaction.
 */
class Transaction {
    /**
     * This constructor takes the values, validates them, assigns them and freezes the object.
     * Use the static factory methods to assist in creating a Transaction object from varying data types.
     * @note Transaction objects implement EIP155 by default. To disable it, pass in an `@ethereumjs/common` object set before EIP155 activation (i.e. before Spurious Dragon).
     */
    constructor(nonce, gasPrice, gasLimit, to, value, data, v, r, s, opts) {
        const validateCannotExceedMaxInteger = {
            nonce,
            gasPrice,
            gasLimit,
            value,
            r,
            s,
        };
        for (const [key, value] of Object.entries(validateCannotExceedMaxInteger)) {
            if (value && value.gt(ethereumjs_util_1.MAX_INTEGER)) {
                throw new Error(`${key} cannot exceed MAX_INTEGER, given ${value}`);
            }
        }
        if (opts === null || opts === void 0 ? void 0 : opts.common) {
            this.common = opts.common;
        }
        else {
            const DEFAULT_CHAIN = 'mainnet';
            this.common = new common_1.default({ chain: DEFAULT_CHAIN });
        }
        this._validateTxV(v);
        this.nonce = nonce;
        this.gasPrice = gasPrice;
        this.gasLimit = gasLimit;
        this.to = to;
        this.value = value;
        this.data = data;
        this.v = v;
        this.r = r;
        this.s = s;
        Object.freeze(this);
    }
    static fromTxData(txData, opts) {
        const { nonce, gasLimit, gasPrice, to, value, data, v, r, s } = txData;
        return new Transaction(new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(nonce)), new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(gasPrice)), new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(gasLimit)), to ? new ethereumjs_util_1.Address(ethereumjs_util_1.toBuffer(to)) : undefined, new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(value)), ethereumjs_util_1.toBuffer(data), new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(v)), new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(r)), new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(s)), opts);
    }
    static fromRlpSerializedTx(serialized, opts) {
        const values = ethereumjs_util_1.rlp.decode(serialized);
        if (!Array.isArray(values)) {
            throw new Error('Invalid serialized tx input. Must be array');
        }
        return this.fromValuesArray(values, opts);
    }
    static fromValuesArray(values, opts) {
        if (values.length !== 6 && values.length !== 9) {
            throw new Error('Invalid transaction. Only expecting 6 values (for unsigned tx) or 9 values (for signed tx).');
        }
        const [nonce, gasPrice, gasLimit, to, value, data, v, r, s] = values;
        return new Transaction(new ethereumjs_util_1.BN(nonce), new ethereumjs_util_1.BN(gasPrice), new ethereumjs_util_1.BN(gasLimit), to && to.length > 0 ? new ethereumjs_util_1.Address(to) : undefined, new ethereumjs_util_1.BN(value), data || buffer_1.Buffer.from([]), v ? new ethereumjs_util_1.BN(v) : undefined, r ? new ethereumjs_util_1.BN(r) : undefined, s ? new ethereumjs_util_1.BN(s) : undefined, opts);
    }
    /**
     * If the tx's `to` is to the creation address
     */
    toCreationAddress() {
        return this.to === undefined || this.to.buf.length === 0;
    }
    /**
     * Computes a sha3-256 hash of the serialized tx
     */
    hash() {
        const values = [
            ethereumjs_util_1.bnToRlp(this.nonce),
            ethereumjs_util_1.bnToRlp(this.gasPrice),
            ethereumjs_util_1.bnToRlp(this.gasLimit),
            this.to !== undefined ? this.to.buf : buffer_1.Buffer.from([]),
            ethereumjs_util_1.bnToRlp(this.value),
            this.data,
            this.v ? ethereumjs_util_1.bnToRlp(this.v) : buffer_1.Buffer.from([]),
            this.r ? ethereumjs_util_1.bnToRlp(this.r) : buffer_1.Buffer.from([]),
            this.s ? ethereumjs_util_1.bnToRlp(this.s) : buffer_1.Buffer.from([]),
        ];
        return ethereumjs_util_1.rlphash(values);
    }
    getMessageToSign() {
        return this._getMessageToSign(this._unsignedTxImplementsEIP155());
    }
    getMessageToVerifySignature() {
        return this._getMessageToSign(this._signedTxImplementsEIP155());
    }
    /**
     * Returns chain ID
     */
    getChainId() {
        return this.common.chainId();
    }
    /**
     * Returns the sender's address
     */
    getSenderAddress() {
        return new ethereumjs_util_1.Address(ethereumjs_util_1.publicToAddress(this.getSenderPublicKey()));
    }
    /**
     * Returns the public key of the sender
     */
    getSenderPublicKey() {
        const msgHash = this.getMessageToVerifySignature();
        // All transaction signatures whose s-value is greater than secp256k1n/2 are considered invalid.
        if (this.common.gteHardfork('homestead') && this.s && this.s.gt(N_DIV_2)) {
            throw new Error('Invalid Signature: s-values greater than secp256k1n/2 are considered invalid');
        }
        const { v, r, s } = this;
        if (!v || !r || !s) {
            throw new Error('Missing values to derive sender public key from signed tx');
        }
        try {
            return ethereumjs_util_1.ecrecover(msgHash, v.toNumber(), ethereumjs_util_1.bnToRlp(r), ethereumjs_util_1.bnToRlp(s), this._signedTxImplementsEIP155() ? this.getChainId() : undefined);
        }
        catch (e) {
            throw new Error('Invalid Signature');
        }
    }
    /**
     * Determines if the signature is valid
     */
    verifySignature() {
        try {
            return ethereumjs_util_1.unpadBuffer(this.getSenderPublicKey()).length !== 0;
        }
        catch (e) {
            return false;
        }
    }
    /**
     * Sign a transaction with a given private key.
     * Returns a new Transaction object (the original tx will not be modified).
     * Example:
     * ```typescript
     * const unsignedTx = Transaction.fromTxData(txData)
     * const signedTx = unsignedTx.sign(privKey)
     * ```
     * @param privateKey Must be 32 bytes in length.
     */
    sign(privateKey) {
        if (privateKey.length !== 32) {
            throw new Error('Private key must be 32 bytes in length.');
        }
        const msgHash = this.getMessageToSign();
        // Only `v` is reassigned.
        /* eslint-disable-next-line prefer-const */
        let { v, r, s } = ethereumjs_util_1.ecsign(msgHash, privateKey);
        if (this._unsignedTxImplementsEIP155()) {
            v += this.getChainId() * 2 + 8;
        }
        const opts = {
            common: this.common,
        };
        return new Transaction(this.nonce, this.gasPrice, this.gasLimit, this.to, this.value, this.data, new ethereumjs_util_1.BN(v), new ethereumjs_util_1.BN(r), new ethereumjs_util_1.BN(s), opts);
    }
    /**
     * The amount of gas paid for the data in this tx
     */
    getDataFee() {
        const txDataZero = this.common.param('gasPrices', 'txDataZero');
        const txDataNonZero = this.common.param('gasPrices', 'txDataNonZero');
        let cost = 0;
        for (let i = 0; i < this.data.length; i++) {
            this.data[i] === 0 ? (cost += txDataZero) : (cost += txDataNonZero);
        }
        return new ethereumjs_util_1.BN(cost);
    }
    /**
     * The minimum amount of gas the tx must have (DataFee + TxFee + Creation Fee)
     */
    getBaseFee() {
        const fee = this.getDataFee().addn(this.common.param('gasPrices', 'tx'));
        if (this.common.gteHardfork('homestead') && this.toCreationAddress()) {
            fee.iaddn(this.common.param('gasPrices', 'txCreation'));
        }
        return fee;
    }
    /**
     * The up front amount that an account must have for this transaction to be valid
     */
    getUpfrontCost() {
        return this.gasLimit.mul(this.gasPrice).add(this.value);
    }
    validate(stringError = false) {
        const errors = [];
        if (!this.verifySignature()) {
            errors.push('Invalid Signature');
        }
        if (this.getBaseFee().gt(this.gasLimit)) {
            errors.push(`gasLimit is too low. given ${this.gasLimit}, need at least ${this.getBaseFee()}`);
        }
        return stringError ? errors : errors.length === 0;
    }
    /**
     * Returns a Buffer Array of the raw Buffers of this transaction, in order.
     */
    raw() {
        return [
            ethereumjs_util_1.bnToRlp(this.nonce),
            ethereumjs_util_1.bnToRlp(this.gasPrice),
            ethereumjs_util_1.bnToRlp(this.gasLimit),
            this.to !== undefined ? this.to.buf : buffer_1.Buffer.from([]),
            ethereumjs_util_1.bnToRlp(this.value),
            this.data,
            this.v !== undefined ? ethereumjs_util_1.bnToRlp(this.v) : buffer_1.Buffer.from([]),
            this.r !== undefined ? ethereumjs_util_1.bnToRlp(this.r) : buffer_1.Buffer.from([]),
            this.s !== undefined ? ethereumjs_util_1.bnToRlp(this.s) : buffer_1.Buffer.from([]),
        ];
    }
    /**
     * Returns the rlp encoding of the transaction.
     */
    serialize() {
        return ethereumjs_util_1.rlp.encode(this.raw());
    }
    /**
     * Returns an object with the JSON representation of the transaction
     */
    toJSON() {
        return {
            nonce: ethereumjs_util_1.bnToHex(this.nonce),
            gasPrice: ethereumjs_util_1.bnToHex(this.gasPrice),
            gasLimit: ethereumjs_util_1.bnToHex(this.gasLimit),
            to: this.to !== undefined ? this.to.toString() : undefined,
            value: ethereumjs_util_1.bnToHex(this.value),
            data: '0x' + this.data.toString('hex'),
            v: this.v !== undefined ? ethereumjs_util_1.bnToHex(this.v) : undefined,
            r: this.r !== undefined ? ethereumjs_util_1.bnToHex(this.r) : undefined,
            s: this.s !== undefined ? ethereumjs_util_1.bnToHex(this.s) : undefined,
        };
    }
    isSigned() {
        const { v, r, s } = this;
        return !!v && !!r && !!s;
    }
    _unsignedTxImplementsEIP155() {
        return this.common.gteHardfork('spuriousDragon');
    }
    _signedTxImplementsEIP155() {
        var _a;
        if (!this.isSigned()) {
            throw Error('This transaction is not signed');
        }
        const onEIP155BlockOrLater = this.common.gteHardfork('spuriousDragon');
        // EIP155 spec:
        // If block.number >= 2,675,000 and v = CHAIN_ID * 2 + 35 or v = CHAIN_ID * 2 + 36, then when computing the hash of a transaction for purposes of signing or recovering, instead of hashing only the first six elements (i.e. nonce, gasprice, startgas, to, value, data), hash nine elements, with v replaced by CHAIN_ID, r = 0 and s = 0.
        const v = (_a = this.v) === null || _a === void 0 ? void 0 : _a.toNumber();
        const vAndChainIdMeetEIP155Conditions = v === this.getChainId() * 2 + 35 || v === this.getChainId() * 2 + 36;
        return vAndChainIdMeetEIP155Conditions && onEIP155BlockOrLater;
    }
    _getMessageToSign(withEIP155) {
        const values = [
            ethereumjs_util_1.bnToRlp(this.nonce),
            ethereumjs_util_1.bnToRlp(this.gasPrice),
            ethereumjs_util_1.bnToRlp(this.gasLimit),
            this.to !== undefined ? this.to.buf : buffer_1.Buffer.from([]),
            ethereumjs_util_1.bnToRlp(this.value),
            this.data,
        ];
        if (withEIP155) {
            values.push(ethereumjs_util_1.toBuffer(this.getChainId()));
            values.push(ethereumjs_util_1.unpadBuffer(ethereumjs_util_1.toBuffer(0)));
            values.push(ethereumjs_util_1.unpadBuffer(ethereumjs_util_1.toBuffer(0)));
        }
        return ethereumjs_util_1.rlphash(values);
    }
    /**
     * Validates tx's `v` value
     */
    _validateTxV(v) {
        if (v === undefined || v.toNumber() === 0) {
            return;
        }
        if (!this.common.gteHardfork('spuriousDragon')) {
            return;
        }
        const vInt = v.toNumber();
        if (vInt === 27 || vInt === 28) {
            return;
        }
        const isValidEIP155V = vInt === this.getChainId() * 2 + 35 || vInt === this.getChainId() * 2 + 36;
        if (!isValidEIP155V) {
            throw new Error(`Incompatible EIP155-based V ${vInt} and chain id ${this.getChainId()}. See the Common parameter of the Transaction constructor to set the chain id.`);
        }
    }
}
exports.default = Transaction;
//# sourceMappingURL=transaction.js.map