"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var ethereumjs_util_1 = require("ethereumjs-util");
var common_1 = __importDefault(require("@ethereumjs/common"));
var legacyTransaction_1 = __importDefault(require("./legacyTransaction"));
var eip2930Transaction_1 = __importDefault(require("./eip2930Transaction"));
var DEFAULT_COMMON = new common_1.default({ chain: 'mainnet' });
var TransactionFactory = /** @class */ (function () {
    // It is not possible to instantiate a TransactionFactory object.
    function TransactionFactory() {
    }
    /**
     * Create a transaction from a `txData` object
     *
     * @param txData - The transaction data. The `type` field will determine which transaction type is returned (if undefined, creates a legacy transaction)
     * @param txOptions - Options to pass on to the constructor of the transaction
     */
    TransactionFactory.fromTxData = function (txData, txOptions) {
        var _a;
        if (txOptions === void 0) { txOptions = {}; }
        var common = (_a = txOptions.common) !== null && _a !== void 0 ? _a : DEFAULT_COMMON;
        if (!('type' in txData) || txData.type === undefined) {
            // Assume legacy transaction
            return legacyTransaction_1.default.fromTxData(txData, txOptions);
        }
        else {
            var txType = new ethereumjs_util_1.BN(ethereumjs_util_1.toBuffer(txData.type)).toNumber();
            return TransactionFactory.getTransactionClass(txType, common).fromTxData(txData, txOptions);
        }
    };
    /**
     * This method tries to decode serialized data.
     *
     * @param data - The data Buffer
     * @param txOptions - The transaction options
     */
    TransactionFactory.fromSerializedData = function (data, txOptions) {
        var _a;
        if (txOptions === void 0) { txOptions = {}; }
        var common = (_a = txOptions.common) !== null && _a !== void 0 ? _a : DEFAULT_COMMON;
        if (data[0] <= 0x7f) {
            // It is an EIP-2718 Typed Transaction
            if (!common.isActivatedEIP(2718)) {
                throw new Error('Common support for TypedTransactions (EIP-2718) not activated');
            }
            // Determine the type.
            var EIP = void 0;
            switch (data[0]) {
                case 1:
                    EIP = 2930;
                    break;
                default:
                    throw new Error("TypedTransaction with ID " + data[0] + " unknown");
            }
            if (!common.isActivatedEIP(EIP)) {
                throw new Error("Cannot create TypedTransaction with ID " + data[0] + ": EIP " + EIP + " not activated");
            }
            return eip2930Transaction_1.default.fromSerializedTx(data, txOptions);
        }
        else {
            return legacyTransaction_1.default.fromSerializedTx(data, txOptions);
        }
    };
    /**
     * When decoding a BlockBody, in the transactions field, a field is either:
     * A Buffer (a TypedTransaction - encoded as TransactionType || rlp(TransactionPayload))
     * A Buffer[] (Legacy Transaction)
     * This method returns the right transaction.
     *
     * @param data - A Buffer or Buffer[]
     * @param txOptions - The transaction options
     */
    TransactionFactory.fromBlockBodyData = function (data, txOptions) {
        if (txOptions === void 0) { txOptions = {}; }
        if (Buffer.isBuffer(data)) {
            return this.fromSerializedData(data, txOptions);
        }
        else if (Array.isArray(data)) {
            // It is a legacy transaction
            return legacyTransaction_1.default.fromValuesArray(data, txOptions);
        }
        else {
            throw new Error('Cannot decode transaction: unknown type input');
        }
    };
    /**
     * This helper method allows one to retrieve the class which matches the transactionID
     * If transactionID is undefined, returns the legacy transaction class.
     *
     * @param transactionID
     * @param common
     */
    TransactionFactory.getTransactionClass = function (transactionID, common) {
        if (transactionID === void 0) { transactionID = 0; }
        var usedCommon = common !== null && common !== void 0 ? common : DEFAULT_COMMON;
        if (transactionID !== 0) {
            if (!usedCommon.isActivatedEIP(2718)) {
                throw new Error('Common support for TypedTransactions (EIP-2718) not activated');
            }
        }
        var legacyTxn = transactionID == 0 || (transactionID >= 0x80 && transactionID <= 0xff);
        if (legacyTxn) {
            return legacyTransaction_1.default;
        }
        switch (transactionID) {
            case 1:
                return eip2930Transaction_1.default;
            default:
                throw new Error("TypedTransaction with ID " + transactionID + " unknown");
        }
    };
    return TransactionFactory;
}());
exports.default = TransactionFactory;
//# sourceMappingURL=transactionFactory.js.map