import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); if (enumerableOnly) symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; }); keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; if (i % 2) { ownKeys(Object(source), true).forEach(function (key) { _defineProperty(target, key, source[key]); }); } else if (Object.getOwnPropertyDescriptors) { Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)); } else { ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } } return target; }

const GLTF_TO_DRACO_ATTRIBUTE_NAME_MAP = {
  POSITION: 'POSITION',
  NORMAL: 'NORMAL',
  COLOR_0: 'COLOR',
  TEXCOORD_0: 'TEX_COORD'
};

function noop() {}

function dracoInt8ArrayToArrayBuffer(dracoData) {
  const byteLength = dracoData.size();
  const outputBuffer = new ArrayBuffer(byteLength);
  const outputData = new Int8Array(outputBuffer);

  for (let i = 0; i < byteLength; ++i) {
    outputData[i] = dracoData.GetValue(i);
  }

  return outputBuffer;
}

export default class DracoBuilder {
  constructor(draco, options = {}) {
    this.draco = draco;
    this.dracoEncoder = new this.draco.Encoder();
    this.dracoMeshBuilder = new this.draco.MeshBuilder();
    this.log = options.log || noop;
  }

  destroy() {
    this.destroyEncodedObject(this.dracoMeshBuilder);
    this.destroyEncodedObject(this.dracoEncoder);
    this.dracoMeshBuilder = null;
    this.dracoEncoder = null;
    this.draco = null;
  }

  destroyEncodedObject(object) {
    if (object) {
      this.draco.destroy(object);
    }
  }

  encodeSync(mesh, options) {
    this._setOptions(options);

    return options.pointcloud ? this._encodePointCloud(mesh) : this._encodeMesh(mesh);
  }

  _getAttributesFromMesh(mesh) {
    const attributes = _objectSpread(_objectSpread({}, mesh), mesh.attributes);

    if (mesh.indices) {
      attributes.indices = mesh.indices;
    }

    return attributes;
  }

  _encodePointCloud(pointcloud) {
    const attributes = this._getAttributesFromMesh(pointcloud);

    const dracoPointCloud = this._createDracoPointCloud(attributes);

    const dracoData = new this.draco.DracoInt8Array();

    try {
      const encodedLen = this.dracoEncoder.EncodePointCloudToDracoBuffer(dracoPointCloud, false, dracoData);

      if (!(encodedLen > 0)) {
        throw new Error('Draco encoding failed.');
      }

      this.log("DRACO encoded ".concat(dracoPointCloud.num_points(), " points\n        with ").concat(dracoPointCloud.num_attributes(), " attributes into ").concat(encodedLen, " bytes"));
      return dracoInt8ArrayToArrayBuffer(dracoData);
    } finally {
      this.destroyEncodedObject(dracoData);
      this.destroyEncodedObject(dracoPointCloud);
    }
  }

  _encodeMesh(mesh) {
    const attributes = this._getAttributesFromMesh(mesh);

    const dracoMesh = this._createDracoMesh(attributes);

    const dracoData = new this.draco.DracoInt8Array();

    try {
      const encodedLen = this.dracoEncoder.EncodeMeshToDracoBuffer(dracoMesh, dracoData);

      if (encodedLen <= 0) {
        throw new Error('Draco encoding failed.');
      }

      this.log("DRACO encoded ".concat(dracoMesh.num_points(), " points\n        with ").concat(dracoMesh.num_attributes(), " attributes into ").concat(encodedLen, " bytes"));
      return dracoInt8ArrayToArrayBuffer(dracoData);
    } finally {
      this.destroyEncodedObject(dracoData);
      this.destroyEncodedObject(dracoMesh);
    }
  }

  _setOptions(options) {
    if ('speed' in options) {
      this.dracoEncoder.SetSpeedOptions(...options.speed);
    }

    if ('method' in options) {
      const dracoMethod = this.draco[options.method];
      this.dracoEncoder.SetEncodingMethod(dracoMethod);
    }

    if ('quantization' in options) {
      for (const attribute in options.quantization) {
        const bits = options.quantization[attribute];
        const dracoPosition = this.draco[attribute];
        this.dracoEncoder.SetAttributeQuantization(dracoPosition, bits);
      }
    }
  }

  _createDracoMesh(attributes) {
    const dracoMesh = new this.draco.Mesh();

    try {
      const positions = this._getPositionAttribute(attributes);

      if (!positions) {
        throw new Error('positions');
      }

      const vertexCount = positions.length / 3;

      for (let attributeName in attributes) {
        const attribute = attributes[attributeName];
        attributeName = GLTF_TO_DRACO_ATTRIBUTE_NAME_MAP[attributeName] || attributeName;

        this._addAttributeToMesh(dracoMesh, attributeName, attribute, vertexCount);
      }
    } catch (error) {
      this.destroyEncodedObject(dracoMesh);
      throw error;
    }

    return dracoMesh;
  }

  _createDracoPointCloud(attributes) {
    const dracoPointCloud = new this.draco.PointCloud();

    try {
      const positions = this._getPositionAttribute(attributes);

      if (!positions) {
        throw new Error('positions');
      }

      const vertexCount = positions.length / 3;

      for (let attributeName in attributes) {
        const attribute = attributes[attributeName];
        attributeName = GLTF_TO_DRACO_ATTRIBUTE_NAME_MAP[attributeName] || attributeName;

        this._addAttributeToMesh(dracoPointCloud, attributeName, attribute, vertexCount);
      }
    } catch (error) {
      this.destroyEncodedObject(dracoPointCloud);
      throw error;
    }

    return dracoPointCloud;
  }

  _addAttributeToMesh(dracoMesh, attributeName, attribute, vertexCount) {
    if (!ArrayBuffer.isView(attribute)) {
      return;
    }

    const dracoAttributeType = this._getDracoAttributeType(attributeName, attribute);

    const size = attribute.length / vertexCount;

    if (dracoAttributeType === 'indices') {
      const numFaces = attribute.length / 3;
      this.log("Adding attribute ".concat(attributeName, ", size ").concat(numFaces));
      this.dracoMeshBuilder.AddFacesToMesh(dracoMesh, numFaces, attribute);
      return;
    }

    this.log("Adding attribute ".concat(attributeName, ", size ").concat(size));

    switch (attribute.constructor.name) {
      case 'Int8Array':
        this.dracoMeshBuilder.AddInt8Attribute(dracoMesh, dracoAttributeType, vertexCount, size, attribute);
        break;

      case 'Int16Array':
        this.dracoMeshBuilder.AddInt16Attribute(dracoMesh, dracoAttributeType, vertexCount, size, attribute);
        break;

      case 'Int32Array':
        this.dracoMeshBuilder.AddInt32Attribute(dracoMesh, dracoAttributeType, vertexCount, size, attribute);
        break;

      case 'Uint8Array':
      case 'Uint8ClampedArray':
        this.dracoMeshBuilder.AddUInt8Attribute(dracoMesh, dracoAttributeType, vertexCount, size, attribute);
        break;

      case 'Uint16Array':
        this.dracoMeshBuilder.AddUInt16Attribute(dracoMesh, dracoAttributeType, vertexCount, size, attribute);
        break;

      case 'Uint32Array':
        this.dracoMeshBuilder.AddUInt32Attribute(dracoMesh, dracoAttributeType, vertexCount, size, attribute);
        break;

      case 'Float32Array':
      default:
        this.dracoMeshBuilder.AddFloatAttribute(dracoMesh, dracoAttributeType, vertexCount, size, attribute);
    }
  }

  _getDracoAttributeType(attributeName, attribute) {
    switch (attributeName.toLowerCase()) {
      case 'indices':
        return 'indices';

      case 'position':
      case 'positions':
      case 'vertices':
        return this.draco.POSITION;

      case 'normal':
      case 'normals':
        return this.draco.NORMAL;

      case 'color':
      case 'colors':
        return this.draco.COLOR;

      case 'texCoord':
      case 'texCoords':
        return this.draco.TEX_COORD;

      default:
        return this.draco.GENERIC;
    }
  }

  _getPositionAttribute(attributes) {
    for (const attributeName in attributes) {
      const attribute = attributes[attributeName];

      const dracoType = this._getDracoAttributeType(attributeName, attribute);

      if (dracoType === this.draco.POSITION) {
        return attribute;
      }
    }

    return null;
  }

}
//# sourceMappingURL=draco-builder.js.map