"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isGLB = isGLB;
exports["default"] = parseGLBSync;

var _loaderUtils = require("@loaders.gl/loader-utils");

var MAGIC_glTF = 0x676c5446;
var GLB_FILE_HEADER_SIZE = 12;
var GLB_CHUNK_HEADER_SIZE = 8;
var GLB_CHUNK_TYPE_JSON = 0x4e4f534a;
var GLB_CHUNK_TYPE_BIN = 0x004e4942;
var GLB_CHUNK_TYPE_JSON_XVIZ_DEPRECATED = 0;
var GLB_CHUNK_TYPE_BIX_XVIZ_DEPRECATED = 1;
var LE = true;

function getMagicString(dataView) {
  var byteOffset = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
  return "".concat(String.fromCharCode(dataView.getUint8(byteOffset + 0))).concat(String.fromCharCode(dataView.getUint8(byteOffset + 1))).concat(String.fromCharCode(dataView.getUint8(byteOffset + 2))).concat(String.fromCharCode(dataView.getUint8(byteOffset + 3)));
}

function isGLB(arrayBuffer) {
  var byteOffset = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
  var options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
  var dataView = new DataView(arrayBuffer);
  var _options$magic = options.magic,
      magic = _options$magic === void 0 ? MAGIC_glTF : _options$magic;
  var magic1 = dataView.getUint32(byteOffset, false);
  return magic1 === magic || magic1 === MAGIC_glTF;
}

function parseGLBSync(glb, arrayBuffer) {
  var byteOffset = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;
  var options = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
  var dataView = new DataView(arrayBuffer);
  glb.type = getMagicString(dataView, byteOffset + 0);
  glb.version = dataView.getUint32(byteOffset + 4, LE);
  var byteLength = dataView.getUint32(byteOffset + 8, LE);
  glb.header = {
    byteOffset: byteOffset,
    byteLength: byteLength
  };
  (0, _loaderUtils.assert)(glb.version === 2, "Invalid GLB version ".concat(glb.version, ". Only supports v2."));
  (0, _loaderUtils.assert)(glb.header.byteLength > GLB_FILE_HEADER_SIZE + GLB_CHUNK_HEADER_SIZE);
  glb.json = {};
  glb.binChunks = [];
  parseGLBChunksSync(glb, dataView, byteOffset + GLB_FILE_HEADER_SIZE, options);
  return byteOffset + glb.header.byteLength;
}

function parseGLBChunksSync(glb, dataView, byteOffset, options) {
  while (byteOffset + 8 <= glb.header.byteLength) {
    var chunkLength = dataView.getUint32(byteOffset + 0, LE);
    var chunkFormat = dataView.getUint32(byteOffset + 4, LE);
    byteOffset += GLB_CHUNK_HEADER_SIZE;

    switch (chunkFormat) {
      case GLB_CHUNK_TYPE_JSON:
        parseJSONChunk(glb, dataView, byteOffset, chunkLength, options);
        break;

      case GLB_CHUNK_TYPE_BIN:
        parseBINChunk(glb, dataView, byteOffset, chunkLength, options);
        break;

      case GLB_CHUNK_TYPE_JSON_XVIZ_DEPRECATED:
        if (!options.glb.strict) {
          parseJSONChunk(glb, dataView, byteOffset, chunkLength, options);
        }

        break;

      case GLB_CHUNK_TYPE_BIX_XVIZ_DEPRECATED:
        if (!options.glb.strict) {
          parseBINChunk(glb, dataView, byteOffset, chunkLength, options);
        }

        break;

      default:
        break;
    }

    byteOffset += (0, _loaderUtils.padTo4Bytes)(chunkLength);
  }

  return byteOffset;
}

function parseJSONChunk(glb, dataView, byteOffset, chunkLength, options) {
  var jsonChunk = new Uint8Array(dataView.buffer, byteOffset, chunkLength);
  var textDecoder = new TextDecoder('utf8');
  var jsonText = textDecoder.decode(jsonChunk);
  glb.json = JSON.parse(jsonText);
}

function parseBINChunk(glb, dataView, byteOffset, chunkLength, options) {
  glb.header.hasBinChunk = true;
  glb.binChunks.push({
    byteOffset: byteOffset,
    byteLength: chunkLength,
    arrayBuffer: dataView.buffer
  });
}
//# sourceMappingURL=parse-glb.js.map