import { getBinaryImageMIMEType, getBinaryImageSize } from '@loaders.gl/images';
import assert from '../utils/assert';
import { UBER_POINT_CLOUD_EXTENSION } from '../gltf-constants';
import GLBBuilder from './glb-builder';

function packBinaryJson(data, builder, packOptions) {
  assert(!packOptions.packTypedArrays);
  return data;
}

export default class GLTFBuilder extends GLBBuilder {
  constructor(options = {}) {
    super(options);
    this.DracoWriter = options.DracoWriter;
    this.DracoLoader = options.DracoLoader;
  }

  addApplicationData(key, data, packOptions = {}) {
    const jsonData = packOptions.packTypedArrays ? packBinaryJson(data, this, packOptions) : data;
    this.json[key] = jsonData;
    return this;
  }

  addExtraData(key, data, packOptions = {}) {
    const packedJson = packOptions.packedTypedArrays ? packBinaryJson(data, this, packOptions) : data;
    this.json.extras = this.json.extras || {};
    this.json.extras[key] = packedJson;
    return this;
  }

  addExtension(extensionName, data, packOptions = {}) {
    assert(data);
    const packedJson = packOptions.packTypedArrays ? packBinaryJson(data, this, packOptions) : data;
    this.json.extensions = this.json.extensions || {};
    this.json.extensions[extensionName] = packedJson;
    this.registerUsedExtension(extensionName);
    return this;
  }

  addRequiredExtension(extensionName, data, packOptions = {}) {
    assert(data);
    const packedJson = packOptions.packTypedArrays ? packBinaryJson(data, this, packOptions) : data;
    this.addExtension(extensionName, packedJson);
    this.registerRequiredExtension(extensionName);
    return this;
  }

  registerUsedExtension(extensionName) {
    this.json.extensionsUsed = this.json.extensionsUsed || [];

    if (!this.json.extensionsUsed.find(ext => ext === extensionName)) {
      this.json.extensionsUsed.push(extensionName);
    }
  }

  registerRequiredExtension(extensionName) {
    this.registerUsedExtension(extensionName);
    this.json.extensionsRequired = this.json.extensionsRequired || [];

    if (!this.json.extensionsRequired.find(ext => ext === extensionName)) {
      this.json.extensionsRequired.push(extensionName);
    }
  }

  addMesh(attributes, indices, mode = 4) {
    assert(false);
    return -1;
  }

  addPointCloud(attributes) {
    assert(false);
    return -1;
  }

  addCompressedMesh(attributes, indices, mode = 4) {
    assert(false);
    return -1;
  }

  addCompressedPointCloud(attributes) {
    if (!this.DracoWriter || !this.DracoLoader) {
      throw new Error('DracoWriter/DracoLoader not available');
    }

    attributes.mode = 0;
    const compressedData = this.DracoWriter.encodeSync(attributes, {
      draco: {
        pointcloud: true
      }
    });
    const bufferViewIndex = this.addBufferView(compressedData);
    const glTFMesh = {
      primitives: [{
        attributes: {},
        mode: 0,
        extensions: {
          [UBER_POINT_CLOUD_EXTENSION]: {
            bufferView: bufferViewIndex
          }
        }
      }]
    };
    this.registerRequiredExtension(UBER_POINT_CLOUD_EXTENSION);
    this.json.meshes = this.json.meshes || [];
    this.json.meshes.push(glTFMesh);
    return this.json.meshes.length - 1;
  }

  addImage(imageData) {
    const bufferViewIndex = this.addBufferView(imageData);
    const mimeType = getBinaryImageMIMEType(imageData) || {};

    if (mimeType) {
      const {
        width,
        height
      } = getBinaryImageSize(imageData, mimeType) || {};
      this.json.images.push({
        bufferView: bufferViewIndex,
        mimeType,
        width,
        height
      });
    } else {
      this.json.images.push({
        bufferView: bufferViewIndex
      });
    }

    return this.json.images.length - 1;
  }

}
//# sourceMappingURL=gltf-builder.js.map