import { DracoLoader } from '@loaders.gl/draco';
import { getZeroOffsetArrayBuffer } from '@loaders.gl/loader-utils';
import GLTFScenegraph from '../gltf-scenegraph';
import { KHR_DRACO_MESH_COMPRESSION } from '../gltf-constants';
import { getGLTFAccessors, getGLTFAccessor } from '../gltf-utils/gltf-attribute-utils';
export async function decode(gltfData, options, context) {
  if (!options.gltf.decompressMeshes) {
    return;
  }

  const scenegraph = new GLTFScenegraph(gltfData);
  const promises = [];

  for (const primitive of meshPrimitiveIterator(scenegraph)) {
    if (scenegraph.getObjectExtension(primitive, KHR_DRACO_MESH_COMPRESSION)) {
      promises.push(decompressPrimitive(primitive, scenegraph, options, context));
    }
  }

  await Promise.all(promises);
  scenegraph.removeExtension(KHR_DRACO_MESH_COMPRESSION);
}
export function encode(gltfData, options = {}) {
  const scenegraph = new GLTFScenegraph(gltfData);

  for (const mesh of scenegraph.json.meshes || []) {
    compressMesh(mesh, options);
    scenegraph.addRequiredExtension(KHR_DRACO_MESH_COMPRESSION);
  }
}

async function decompressPrimitive(primitive, scenegraph, options, context) {
  const compressedPrimitive = scenegraph.getObjectExtension(primitive, KHR_DRACO_MESH_COMPRESSION);
  const buffer = scenegraph.getTypedArrayForBufferView(compressedPrimitive.bufferView);
  const bufferCopy = getZeroOffsetArrayBuffer(buffer.buffer, buffer.byteOffset);
  const {
    parse
  } = context;
  const decodedData = await parse(bufferCopy, DracoLoader, options, context);
  primitive.attributes = getGLTFAccessors(decodedData.attributes);

  if (decodedData.indices) {
    primitive.indices = getGLTFAccessor(decodedData.indices);
  }

  checkPrimitive(primitive);
}

function compressMesh(attributes, indices, mode = 4, options, context) {
  if (!options.DracoWriter || !options.DracoLoader) {
    throw new Error('DracoWriter/DracoLoader not available');
  }

  const compressedData = options.DracoWriter.encodeSync({
    attributes
  });
  const {
    parseSync
  } = context;
  const decodedData = parseSync({
    attributes
  });

  const fauxAccessors = options._addFauxAttributes(decodedData.attributes);

  const bufferViewIndex = options.addBufferView(compressedData);
  const glTFMesh = {
    primitives: [{
      attributes: fauxAccessors,
      mode,
      extensions: {
        [KHR_DRACO_MESH_COMPRESSION]: {
          bufferView: bufferViewIndex,
          attributes: fauxAccessors
        }
      }
    }]
  };
  return glTFMesh;
}

function checkPrimitive(primitive) {
  if (!primitive.attributes && Object.keys(primitive.attributes).length > 0) {
    throw new Error('Empty glTF primitive detected: Draco decompression failure?');
  }
}

function* meshPrimitiveIterator(scenegraph) {
  for (const mesh of scenegraph.json.meshes || []) {
    for (const primitive of mesh.primitives) {
      yield primitive;
    }
  }
}
//# sourceMappingURL=KHR_draco_mesh_compression.js.map